// Lab three is part of the "lab3" package
package sokoban;

// Some handy includes
import java.awt.*;
import java.awt.geom.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import java.awt.Event;
import java.awt.event.*;
import sokoban.*;

//
// This is the "controller" of the MVC design layout
//
public class sApp extends JFrame implements ActionListener
{
    // Model and view objects
    private sBoard sboard;
    private sView  sview;
    private sScore sscore;

    // Some event/interface state
    private boolean drag;
    
    ///////////////////////////////////////////////////////////////////////////////
    //                       Initialization (constructor)
    ///////////////////////////////////////////////////////////////////////////////

    // Constructor for the sApp class
    public sApp() 
    {
	// Do some basic initializaton
	super("Sokoban - Aaron Vose");
	setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

	// Create a model (board) object
	sboard = new sBoard();

	// Create and add the score board to our window
	sscore = new sScore();
	getContentPane().add(sscore, BorderLayout.PAGE_END);

	// Create and add the view object to our window
	sview = new sView(sboard);
	getContentPane().add(sview, BorderLayout.CENTER);

	// Create and add a menu bar and menu
	JMenuBar menuBar = new JMenuBar();
	JMenu menu = new JMenu("Actions");
	menu.setMnemonic(KeyEvent.VK_A);
	menuBar.add(menu);

	// Add items to the menu
	JMenuItem menuItem = new JMenuItem("Reset Game", KeyEvent.VK_R);
	menuItem.addActionListener(this);
	menu.add(menuItem);
	menuItem = new JMenuItem("Load Board", KeyEvent.VK_L);
	menuItem.addActionListener(this);
	menu.add(menuItem);
	menuItem = new JMenuItem("Quit Game",  KeyEvent.VK_Q);
	menuItem.addActionListener(this);
	menu.add(menuItem);

	// Intall the menu bar in the jframe
	this.setJMenuBar(menuBar);

	///////////////////////////  Mouse Button Callback //////////////////////////

	// Install a mouse listener for mouse butten events
	addMouseListener(new MouseAdapter() 
	    {
		public void mouseReleased(MouseEvent e) {
		    if( e.getButton() == MouseEvent.BUTTON1 )
			drag = false;
		}
		public void mousePressed(MouseEvent e) {
		    switch(e.getButton()) {
		    case MouseEvent.BUTTON3:
			// Right click is the undo button
			int uc;

			drag = false;
			if ( (uc=sboard.undo()) < 0 ) {
			    sscore.inc_pushes(-1);
			    sscore.inc_moves(uc);
			    repaint();
			} else if(uc > 0) {
			    sscore.inc_moves(-uc);
			    repaint();
			}
			break;
		    case MouseEvent.BUTTON1:
			// Normal left click
			double w  = sview.getWidth();
			double h  = sview.getHeight();
			double ox = (w/2)-(sboard.getWidth()*sView.SIZE)/2;
			double oy = (h/2)-(sboard.getHeight()*sView.SIZE)/2;
			int    ex = e.getX() - (sview.getLocationOnScreen().x-getX());
			int    ey = e.getY() - (sview.getLocationOnScreen().y-getY());
			
			// Check for bounds
			if( (ex < ox) || (ex > ox+(sboard.getWidth()*sView.SIZE)) )  return;
			if( (ey < oy) || (ey > oy+(sboard.getHeight()*sView.SIZE)) ) return;
			
			// Compute board coordinates
			int x = (int)((ex-ox)/sView.SIZE);
			int y = (int)((ey-oy)/sView.SIZE);
			
			// See if the user clicked on the person
			if( (x == sboard.getX()) && (y == sboard.getY()) ) {
			    drag = true;
			    return;
			}
			
			// Not out of bounds, not the person.  Empty space or ball?
			switch(sboard.getSquare(x,y)) {
			case sBoard.GOAL:
			case sBoard.EMPTY:
			    // Empty square was clicked on, try a warp
			    int mvs;
			    if( (mvs=sboard.warp(x,y)) > 0 ) {
				sscore.inc_moves(mvs);
				drag = true;
				repaint();
			    } else {
				Toolkit.getDefaultToolkit().beep();
			    }
			    break;
			case sBoard.BALL:
			    // User clicked on an adjacent ball; this is (probably) a push request
			    if( (Math.abs(sboard.getX()-x) < 2) && (Math.abs(sboard.getY()-y) < 2) )
				tryMove(x, y);
			    break;
			}
		    default:
			// Anything for the other mouse buttons?
			break;
		    }
		}
	    });

	///////////////////////////  Mouse Move Callback //////////////////////////

	// Add a mouse listener for mouse motion
	addMouseMotionListener(new MouseMotionAdapter() 
	    {
		public void mouseDragged(MouseEvent e) {
		    if (drag) {
			double w  = sview.getWidth();
			double h  = sview.getHeight();
			double ox = (w/2)-(sboard.getWidth()*sView.SIZE)/2;
			double oy = (h/2)-(sboard.getHeight()*sView.SIZE)/2;
			int    ex = e.getX() - (sview.getLocationOnScreen().x-getX());
			int    ey = e.getY() - (sview.getLocationOnScreen().y-getY());
			
			// Check for bounds
			if( (ex < ox) || (ex > ox+(sboard.getWidth()*sView.SIZE)) )  return;
			if( (ey < oy) || (ey > oy+(sboard.getHeight()*sView.SIZE)) ) return;
			
			// Compute board coordinates
			int x = (int)((ex-ox)/sView.SIZE);
			int y = (int)((ey-oy)/sView.SIZE);
			
			// Alright, try to do a move.
			tryMove(x, y);
		    }
		}
	    });
	
	////////////////////////////  Key Press Callback ///////////////////////////

	// Add a keyboard listener
	addKeyListener(new KeyAdapter()
	    {
		public void keyPressed(KeyEvent e) {
		    switch(e.getKeyCode()) {
		    case KeyEvent.VK_H:
		    case KeyEvent.VK_LEFT:
			personMoved( sboard.movePerson(sBoard.LEFT) );
			break;
		    case KeyEvent.VK_L:
		    case KeyEvent.VK_RIGHT:
			personMoved( sboard.movePerson(sBoard.RIGHT) );
			break;
		    case KeyEvent.VK_K:
		    case KeyEvent.VK_UP:
			personMoved( sboard.movePerson(sBoard.UP) );
			break;
		    case KeyEvent.VK_J:
		    case KeyEvent.VK_DOWN:
			personMoved( sboard.movePerson(sBoard.DOWN) );
			break;
		    }
		}
	    });
    }

    ///////////////////////////////////////////////////////////////////////////////
    //                     Movement Facilitation Functions
    ///////////////////////////////////////////////////////////////////////////////

    // Tries to apply a move, beeps and clears drag on failure, and repaints on success
    private void tryMove(int x, int y)
    {
	switch(sboard.getX()-x) {
	case 0: break;
	case 1: // Left
	    if( personMoved( sboard.movePerson(sBoard.LEFT) ) < 1 ) {
		drag = false;
		Toolkit.getDefaultToolkit().beep();
	    }
	    break;
	case -1: // Right
	    if( personMoved( sboard.movePerson(sBoard.RIGHT) ) < 1 ) {
		drag = false;
		Toolkit.getDefaultToolkit().beep();
	    }
	    break;			    
	}
	switch(sboard.getY()-y) {
	case 0: break;
	case 1: // Up
	    if( personMoved( sboard.movePerson(sBoard.UP) ) < 1 ) {
		drag = false;
		Toolkit.getDefaultToolkit().beep();
	    }
	    break;
	case -1: // Down
	    if( personMoved( sboard.movePerson(sBoard.DOWN) ) < 1 ) {
		drag = false;
		Toolkit.getDefaultToolkit().beep();
	    }
	    break;			    
	}
    }

    // This function does almost everything that needs to happen on a sucessful move event
    private int personMoved(int rv)
    {
	// Skip if move wasn't valid
	if( rv < 1 ) return rv;

	// Increment move count and push count as needed
	sscore.inc_moves(1);
	if ( rv == 2 ) sscore.inc_pushes(1);

	// Ask for a repaint
	repaint();

	// Check for the win condition
	if(sboard.isWon()) {
	    drag = false;
	    JOptionPane.showMessageDialog(this, "You beat the game in " + sscore.getMoves() + " moves and " + 
					  sscore.getPushes() + " pushes!", "You win!",
					  JOptionPane.WARNING_MESSAGE);
	}

	// Pass along the return value
	return rv;
    }

    ///////////////////////////////////////////////////////////////////////////////
    //                    main() and the Action Event Handler
    ///////////////////////////////////////////////////////////////////////////////

    // This is the callback for the app's menu
    public void actionPerformed(ActionEvent e) 
    {
        JMenuItem source = (JMenuItem)(e.getSource());

        if(source.getText().equals("Quit Game")) {
	    // Exit
	    System.exit(0);
	}
        if(source.getText().equals("Reset Game")) {
	    // Reset the the game state
	    sboard.reset();
	    sscore.reset();
	    repaint();
	    return;
	}
        if(source.getText().equals("Load Board")) {
	    // Load a new board
	    FileDialog ld = new FileDialog(this, "Select game board file", FileDialog.LOAD);
	    ld.setVisible(true);
	    if( ld.getFile() != null )
		sboard.load(ld.getFile());
	    sscore.reset();
	    pack();
	    repaint();
	    return;
	}
    }

    // Application entry point:
    public static void main(String args[])
    {
	sApp sokoban;
	
	// Create setup and display the app
	sokoban = new sApp();
	sokoban.pack();
	sokoban.setVisible(true);
    }
}
