// Lab three is part of the "lab3" package
package sokoban;

// Some handy includes
import java.awt.*;
import java.awt.geom.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import java.awt.Event;
import java.awt.event.*;
import sokoban.*;

//
// This is the "model" of the MVC design layout
//
public class sBoard
{
    // The name of the file this board object represents
    private String fn = null;

    // Board state
    private StringBuffer board[];
    private int          width;
    private int          height;
    private int          px,py;

    // Some constants
    public static final int EMPTY = 0;
    public static final int BALL  = 1;
    public static final int WALL  = 2;
    public static final int GOAL  = 3;

    public static final int UP    = 0;
    public static final int DOWN  = 1;
    public static final int LEFT  = 2;
    public static final int RIGHT = 3;

    // Constructor for this class
    public sBoard() 
    {
    }
    
    ///////////////////////////////////////////////////////////////////////////////
    //                    This code helps the undo feature
    ///////////////////////////////////////////////////////////////////////////////

    // Stack to keep track of moves for the undo feature, and a move struct
    // which will be used with the stack to provide an undo history.
    private Stack        moves;

    public class move {
	int     ox,oy;
	boolean b;
	int     bx,by;
	int     mc;
	public move(int aox, int aoy, boolean ab, int abx, int aby, int amc) 
	{ 
	    ox = aox; oy = aoy; b = ab; bx = abx; by = aby; mc = amc;
	}
    }

    ///////////////////////////////////////////////////////////////////////////////
    //               This code helps the shortest path algorithm
    ///////////////////////////////////////////////////////////////////////////////
    
    // This class is really just used as a C struct
    // and will aide the shortest path algorithm.
    public class square {
	int     x, y, l;
	public square(int ax, int ay, int al) 
	{ 
	    x = ax; y = ay; l = al;
	}
    }

    // Returns true if (x,y) is a position that is valid 
    // to be considered by the shortest path algorithm.
    private boolean spValid(int x, int y, square[][] b) 
    {
	return ( (x > -1) && (x < width) && (y > -1) && (y < height)       &&    // In bounds
		 ( (getSquare(x,y) == EMPTY) || (getSquare(x,y) == GOAL) ) &&    // Empty
		 (b[x][y].l == -1)                                            ); // Not marked
    }

    ///////////////////////////////////////////////////////////////////////////////
    //                           State Read members
    ///////////////////////////////////////////////////////////////////////////////

    // Returns true if the game is won
    public boolean isWon()
    {
	int i,j;

	for(i=0; i<height; i++)
	    for(j=0; j<board[i].length(); j++)
		if( board[i].charAt(j) == '.' )
		    return false;
	return true;
    }

    // Returns the person's x coord
    public int getX()
    {
	return px;
    }
    
    // Returns the person's x coord
    public int getY()
    {
	return py;
    }

    // Returns the valid state of the board object
    public boolean isValid()
    {
	return ( fn != null );
    }

    // Returns the (maximum) width of the board
    public int getWidth() 
    {
	return width;
    }

    // Returns the height of the board
    public int getHeight() 
    {
	return height;
    }

    // This method returns the contents of the square at (x,y)
    public int getSquare(int x, int y)
    {
	// Check for some special cases
	if( fn == null )             return EMPTY;
	if( y >= height )            return EMPTY;
	if( y <  0 )                 return EMPTY;
	if( x <  0 )                 return EMPTY;
	if( x >= board[y].length() ) return EMPTY;

	// Return the contents of (x,y)
	switch(board[y].charAt(x)) {
	case '#': return WALL;
	case '*': // BALL && GOAL
	case '$': return BALL;
	case '.': return GOAL;
	}
	return EMPTY;
    }

    ///////////////////////////////////////////////////////////////////////////////
    //                        State modification members
    ///////////////////////////////////////////////////////////////////////////////

    // This method returns the contents of the square at (x,y)
    private void setSquare(int x, int y, int type)
    {
	// Check for some special cases
	if( fn == null )             return;
	if( y >= height )            return;
	if( y <  0 )                 return;
	if( x <  0 )                 return;
	if( x >= board[y].length() ) return;

	// Return the contents of (x,y)
	switch(type) {
	case WALL: 
	    board[y].setCharAt(x,'#'); 
	    break;
	case BALL:
	    if(board[y].charAt(x) == '.') board[y].setCharAt(x,'*');
	    else                          board[y].setCharAt(x,'$');
	    break;
	case GOAL:
	    board[y].setCharAt(x,'.');
	    break;
	case EMPTY:
	    if(board[y].charAt(x) == '*') board[y].setCharAt(x,'.');
	    else                          board[y].setCharAt(x,' ');
	    break;
	}
    }

    // Moves the person and any balls in the way in direction 'dir' if possible.
    // It returns true if the move was successfull.  Also stores the move in
    // the history stack.
    public int movePerson(int dir)
    {
	boolean push = false;
	int     nx,ny,ex,ey;

	if( fn == null ) 
	    return 0;

	// I think this bit-twiddling is pretty neat actually.
        // Here I apply 'dir' to the current person position to
	// calculate the new and extended (two-steps) positions
	// without a signle branch statement!
	nx  = px + ( (dir>>1)     & (dir&1)     );
	nx -=      ( (dir>>1)     & ((px+1)-nx) );
	ny  = py + ( ((dir>>1)^1) & (dir&1)     );
	ny -=      ( ((dir>>1)^1) & ((py+1)-ny) );

	ex  = nx + ( (dir>>1)     & (dir&1)     );
	ex -=      ( (dir>>1)     & ((nx+1)-ex) );
	ey  = ny + ( ((dir>>1)^1) & (dir&1)     );
	ey -=      ( ((dir>>1)^1) & ((ny+1)-ey) );

	// Check to see if the new position is valid
	if( (ny >= height) || (ny <  0) || (nx <  0) || (nx >= board[ny].length()) || 
	    ( (getSquare(nx,ny) != EMPTY) && (getSquare(nx,ny) != GOAL) && (getSquare(nx,ny) != BALL) ) )
	    return 0;
	if( (getSquare(nx,ny) == BALL) &&
	    ( (ey >= height ) || (ey <  0) || (ex < 0) || (ex >= board[ey].length()) || 
	     ( (getSquare(ex,ey) != EMPTY) && (getSquare(ex,ey) != GOAL)) ) )
	    return 0;

	// Check to see if we need to push a ball
	if( getSquare(nx,ny) == BALL ) {
	    setSquare(nx,ny,EMPTY);
	    setSquare(ex,ey,BALL);

	    push = true;
	}

	// Push a trace of the move we just did onto the
	// move stack to facilitate the undo history.
	moves.push(new move(px, py, push, ex, ey, 1));

	// Actually set the person to their new position
	px = nx;
	py = ny;

	// Return success
	if(push) return 2;
	return 1;
    }

    // Moves the person to (x,y) by means of the shortest path
    // if said path exists.  The return value is the number of
    // moves taken, 0 indicates failure.
    public int warp(int x, int y)
    {
	int    i,j;
	square s;
	
	// Clone the board so we have some memeory to work with.
	// ("board" is not so easy to work with as an array of StringBuffers)
	square b[][] = new square[width][height];
	for(i=0; i<width; i++)
	    for(j=0; j<height; j++)
		b[i][j] = new square(i,j,-1);
	
	// Create a FIFO to help with a bredth-first traversal of the graph.
	LinkedList fifo = new LinkedList();
	
	// Alright, let's start the shortest path algorithm:
	// Place the starting square on the FIFO
	b[px][py].l = 0;
	fifo.addFirst(b[px][py]);
	while(true) {
	    try {
		// Take item off the FIFO, label and add children
		// to the FIFO if they exist and are not labeled.
		s = (square) fifo.removeLast();
		// Left
		if( spValid(s.x-1, s.y, b) ) {
		    b[s.x-1][s.y].l = s.l+1;
		    fifo.addFirst(b[s.x-1][s.y]);
		}
		// Right
		if( spValid(s.x+1, s.y, b) ) {
		    b[s.x+1][s.y].l = s.l+1;
		    fifo.addFirst(b[s.x+1][s.y]);
		}
		// Up
		if( spValid(s.x, s.y-1, b) ) {
		    b[s.x][s.y-1].l = s.l+1;
		    fifo.addFirst(b[s.x][s.y-1]);
		}
		// Down
		if( spValid(s.x, s.y+1, b) ) {
		    b[s.x][s.y+1].l = s.l+1;
		    fifo.addFirst(b[s.x][s.y+1]);
		}
	    } catch(NoSuchElementException e) {
		// This should catch a "NoSuchElementException"
		// when the FIFO becomes empty.  This is the loop
		// termination contdition.
		break;
	    }
	}
	
	// The desired end-point will now be marked with the lowest
	// posible distance from the start point, or -1 if no path 
	// exists.
	if( b[x][y].l == -1 ) 
	    return 0;
	
	// Push a trace of the move we just did onto the
	// move stack to facilitate the undo history.
	moves.push(new move(px, py, false, 0, 0, b[x][y].l));

	// Alright! We found the cost of the shortest path.  Do the warp!
	px = x;
	py = y;
	return b[x][y].l;
    }

    // This will attempt an undo, and return an integer which 
    // acts as both a sucess flag and an move count:
    //  0: failure
    // -x: x moves were undone, 1 push was undone
    //  x: x moves were undone, no pushes were undone
    public int undo()
    {	
	if( fn == null ) return 0;

	try {
	    move m = (move) moves.pop();
	    if(m.b) {
		setSquare(px,py,BALL);
		setSquare(m.bx,m.by,EMPTY);
	    }
	    px = m.ox;
	    py = m.oy;
	    if( m.b ) return -m.mc;
	    return m.mc;
	} catch(Exception ex) {
	}

	return 0;
    }

    ///////////////////////////////////////////////////////////////////////////////
    //                         State Load/Reset members
    ///////////////////////////////////////////////////////////////////////////////

    // This method loads a board file
    public void load(String afn)
    {
	// Save the file name
	fn = afn;
	moves = new Stack();
	
	// Load the board file
	try {
	    BufferedReader in = new BufferedReader(new FileReader(fn));
	    String         line;
	    int            i,j;

	    // Count the number of lines in the .skb file
	    for(height=0; in.readLine() != null; height++ );
	    in.close();

	    // Read in the file
	    board = new StringBuffer[height];
	    in = new BufferedReader(new FileReader(fn));
	    for(i=width=0; (line=in.readLine()) != null; i++) {
		board[i] = new StringBuffer(line);
		// Also track the longets line at the same time
		if(board[i].length() > width) 
		    width = board[i].length();
		// While we are at it, let's look for the '@' marker.
		for(j=0; j<board[i].length(); j++) {
		    if(line.charAt(j) == '@') {
			py = i;
			px = j;
			board[i].setCharAt(j, ' ');
		    }
		}
	    }
	    in.close();
	} catch (Exception e) { 
	    System.err.println("Input file error: " + fn);
	    System.exit(1);
	}
    }

    // This method resets the board to it's starting state
    public void reset()
    {
	if( fn == null ) return;
	load(fn);
    }
}
