#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <math.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/select.h>
#include <pthread.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <GL/glx.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "ptree.h"

/***************************************************************************************
 * Gobals
 ***************************************************************************************/

// Process Tree
pTreen  Root;
int     MaxPid;
int     MaxDepth;
pTreen  Selected;
pTreen  Hover;

// Thread sync
pthread_mutex_t TreeMutex;

// Window and mouse information
int     Width=800,Height=600;
float   Rot[2], Trans[3];      // Translation and rotation values
int     Mx=10000,My=10000;     // Mouse coordinates
char    User[64];              // Current user

// GL matricies
GLint   BaseViewport[4];  
double  BaseProjection[16];
double  BaseModelView[16];

/***************************************************************************************
 * Prototypes
 ***************************************************************************************/

static void   DrawScene(GLWindow *glw);
static pTreen SearchTree(pTreen n, int pid);

/***************************************************************************************
 * Font / GL / X / etc
 ***************************************************************************************/

static void BuildFont(GLWindow *glw, char *fname, pFont f)
{
  XFontStruct *font;  GLuint fbase;   
    
  /* Storage for 96 characters */
  fbase = glGenLists(96);      

  /* Load a font with a specific name in "Host Portable Character Encoding" */
  if ( !(font = XLoadQueryFont(glw->dpy, fname)) ) {
    fprintf(stderr,"Could not load font \"%s\"\n",fname);
    exit(-1);
  }
  f->h = font->ascent + font->descent;
  f->w = font->max_bounds.width;

  /* Build 96 display lists out of our font starting at char 32 */
  glXUseXFont(font->fid, 32, 96, fbase);
  XFreeFont(glw->dpy, font);
  f->f = fbase;
}

static void DeleteFont(pFont f)
{
  glDeleteLists(f->f, 96);
}

// Like printf, but for OpenGL
static void printGLf(pFont f, const char *fmt, ...)
{
  va_list ap;  char text[256];

  if (fmt == NULL) return;

  va_start(ap, fmt);  
  vsprintf(text, fmt, ap);
  va_end(ap);
  glPushAttrib(GL_LIST_BIT);
  glListBase(f->f - 32);
  glCallLists(strlen(text), GL_UNSIGNED_BYTE, text);
  glPopAttrib();
}

// Called when window is resized
static void ResizeGLScene(unsigned int width, unsigned int height)
{
  glViewport(0, 0, width, height); 
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(55.0f,(GLfloat)width/(GLfloat)height,0.1f,2000.0f);
  glTranslatef(0.0f, 0.0f, -80.0f);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
}

// Kills an OpenGL window
static void KillGLWindow(GLWindow *glw)
{
  if (glw->ctx) {
    glDeleteLists(0, 65535);
    if (!glXMakeCurrent(glw->dpy, None, NULL))
      printf("Could not release drawing context.\n");
    glXDestroyContext(glw->dpy, glw->ctx);
    glw->ctx = NULL;
  }
  XCloseDisplay(glw->dpy);
}

// Creates an OpenGL enabled window
static void CreateGLWindow(GLWindow *glw, char* title, int width, int height)
{
  XVisualInfo *vi;  Colormap cmap;  Window twin;  Atom wmDelete;  int glxM, glxm, t;
    
  // Connect to X
  if( !(glw->dpy    = XOpenDisplay(0)) ) {
    fprintf(stderr,"Cannot connect to X server!\n");
    exit(1);
  }
  glw->screen = DefaultScreen(glw->dpy);

  // Get the appropriate visual
  glXQueryVersion(glw->dpy, &glxM, &glxm);
  printf("%d: glX-Version %d.%d\n", glw->id, glxM, glxm);
  if ( (vi = glXChooseVisual(glw->dpy, glw->screen, attrListDbl)) ) {
    printf("%d: Selected doublebuffered mode.\n", glw->id);
  } else {
    vi = glXChooseVisual(glw->dpy, glw->screen, attrListSgl);
    printf("%d: Selected singlebuffered mode.\n", glw->id);
  }

  // Create a GLX context and color map
  glw->ctx               = glXCreateContext(glw->dpy, vi, 0, GL_TRUE);
  cmap                   = XCreateColormap(glw->dpy, RootWindow(glw->dpy, vi->screen), vi->visual, AllocNone);
  glw->attr.colormap     = cmap;
  glw->attr.border_pixel = 0;

  // Create the window
  glw->attr.event_mask = ExposureMask | KeyPressMask | ButtonPressMask| ButtonReleaseMask | StructureNotifyMask | PointerMotionMask;
  glw->win = XCreateWindow(glw->dpy, RootWindow(glw->dpy, vi->screen),
                           0, 0, width, height, 0, vi->depth, InputOutput, vi->visual,
                           CWBorderPixel | CWColormap | CWEventMask, &glw->attr);
  wmDelete = XInternAtom(glw->dpy, "WM_DELETE_WINDOW", True);
  XSetWMProtocols(glw->dpy, glw->win, &wmDelete, 1);
  XSetStandardProperties(glw->dpy, glw->win, title, title, None, NULL, 0, NULL);
  XMapRaised(glw->dpy, glw->win);
    
  // Connect the glx-context to the window
  glXMakeCurrent(glw->dpy, glw->win, glw->ctx);
  XGetGeometry(glw->dpy, glw->win, &twin, &glw->x, &glw->y,
               &glw->width, &glw->height, (unsigned int*)&t, &glw->depth);
  printf("%d: Depth %d\n", glw->id, glw->depth);
  if (glXIsDirect(glw->dpy, glw->ctx)) printf("%d: Direct Rendering enabled.\n", glw->id);
  else                                 printf("%d: Direct Rendering disabled.\n", glw->id);
}

static void yeild(int us)
{
  struct timeval tv;

  // Sleep for a ms
  tv.tv_sec  = 0;
  tv.tv_usec = us;
  select(0, NULL,  NULL, NULL, &tv);
}

static int AnyEvent(Display *d, XEvent *e, XPointer p)
{
  return True;
}

/***************************************************************************************
 * Util functions used by drawing code
 ***************************************************************************************/

static void InitGLWindow(GLWindow *glw)
{
  glShadeModel(GL_SMOOTH);
  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
  glClearDepth(1.0f);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  BuildFont(glw, "fixed", &glw->font);   
  ResizeGLScene(glw->width, glw->height);
  glFlush();
}

static void glOutlineCircle(int x, int y, double r)
{
  int k;

  glBegin(GL_LINE_LOOP);
  for(k=0;k<32;k++)
    glVertex2d(x+r*cos(k*2*M_PI/32),y+r*sin(k*2*M_PI/32));
  glEnd();
}

static void glCircle(int x, int y, double r)
{
  int k;

  glBegin(GL_POLYGON);
  for(k=0;k<32;k++)
    glVertex2d(x+r*cos(k*2*M_PI/32),y+r*sin(k*2*M_PI/32));
  glEnd();
}

/***************************************************************************************
 * Tree drawing code and helpers
 ***************************************************************************************/

static void pColor(int pid)
{
  // Builds a unique color from a pid
  //glColor3f(pid/((float)MaxPid), 1-pid/((float)MaxPid), pid/((float)MaxPid));
  glColor3f(0.0f, 1.0f, 0.0f);
}

static void lColor(int l)
{
  // Builds a unique color from a level
  glColor3f(l/((float)MaxDepth), 0.0f, 1-l/((float)MaxDepth));
}

static void DrawTreeRecursive(GLWindow *glw, pTreen n)
{
  int i,sl,ns,cpz;  char prefix[128];  double x,y,z;

  // Draw non selected items
  if( !n->s ) {
    // Normal node display
    if( !strcmp(n->o,"root") )    glColor3f(1.0f, 0.0f, 0.0f);
    else if( !strcmp(n->o,User) ) glColor3f(0.0f, 1.0f, 0.0f);
    else                          glColor3f(1.0f, 0.8f, 0.0f);
    // Trim off first word into prefix
    for(i=0,sl=strlen(n->n); (i < sl) && (n->n[i] != ' '); i++);
    if( i > sizeof(prefix) ) {
      memcpy(prefix, n->n, sizeof(prefix));
      prefix[sizeof(prefix)-1] = 0;
    } else {
      sscanf(n->n, "%s", prefix);
    }
    // Hover Mode gets full name as prefix
    if( n == Hover ) { 
      cpz = ( (sizeof(prefix) < strlen(n->n)+1) ? (sizeof(prefix)) : (strlen(n->n)+1) );
      memcpy(prefix, n->n, cpz);
      prefix[cpz] = 0;
      glColor3f(1.0f, 1.0f, 0.0f);
    }

    // Project to 2d
    gluProject(n->l[0], n->l[1], n->l[2], BaseModelView, BaseProjection, BaseViewport, &x, &y, &z);

    // Change to 2d
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    glOrtho(0., Width, Height, 0., 1, -100);
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    sl = glw->font.w*strlen(prefix);

    // Draw name here
    glLoadName(n->pid);
    glRasterPos2f(x - sl/2., Height-y-5);
    printGLf(&glw->font, "%s", prefix);
    glLoadName(-1);

    // Draw node markers
    if( n->m ) {
      glColor3f(1.0f, 1.0f, 0.5f);
      ns = 1;
    } else {
      ns = 3;
      glColor3f(0.0f, 1.0f, 1.0f);
    }
    glDisable(GL_DEPTH_TEST);
    glBegin(GL_QUADS);
    glVertex3f(x + -ns, Height-y+ns, 0.0f);
    glVertex3f(x +  ns, Height-y+ns, 0.0f);
    glVertex3f(x +  ns, Height-y-ns, 0.0f);
    glVertex3f(x + -ns, Height-y-ns, 0.0f);
    glEnd();  
    glEnable(GL_DEPTH_TEST);

    // Back to 3d
    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
  }

  // Draw a line from ourselves to each of our children,
  // as we let our children draw themselves.
  for(i=0; i<n->nc; i++) {
    // Line
    glBegin(GL_LINES);
    lColor(n->d);
    glVertex3f(n->l[0], n->l[1], n->l[2]);
    lColor(n->c[i]->d);
    glVertex3f(n->c[i]->l[0], n->c[i]->l[1], n->c[i]->l[2]);
    glEnd();  
    // Child
    DrawTreeRecursive(glw, n->c[i]);
  } 
}

static void DrawBoxesRecursive(GLWindow *glw, pTreen n)
{
  int    i,sl;  double x,y,z;

  // Draw ourself
  if( n->s ) {
    // Selected display
    // Get screen coords
    gluProject(n->l[0], n->l[1], n->l[2], BaseModelView, BaseProjection, BaseViewport, &x, &y, &z);

    // Chnage to 2d no depth
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    glOrtho(0., Width, Height, 0., 1, -100);
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glDisable(GL_DEPTH_TEST);
    sl = glw->font.w*strlen(n->n);
    if( sl < 64 ) sl = 64;

    // Draw translucent nametag background
    glColor3f(0.0f, 1.0f, 1.0f);
    glLoadName(n->pid);
    glColor4f(0.0f, 0.0f, 0.0f, 0.4f);
    glBegin(GL_QUADS);
    glVertex3f(x - sl/2.-16, Height-y+34, 0.0f);
    glVertex3f(x + sl/2.+16, Height-y+34, 0.0f);
    glVertex3f(x + sl/2.+16, Height-y-34, 0.0f);
    glVertex3f(x - sl/2.-16, Height-y-34, 0.0f);
    glEnd();
    
    // Draw border
    if( (abs(x-Mx) < (sl/2.+16)) && (abs((Height-y)-My) < 32) ) glColor3f(1.0f, 1.0f, 0.0f);
    else                                                        glColor3f(1.0f, 1.0f, 1.0f);
    glBegin(GL_LINE_LOOP);
    glVertex3f(x - sl/2.-16, Height-y+34, 0.0f);
    glVertex3f(x + sl/2.+16, Height-y+34, 0.0f);
    glVertex3f(x + sl/2.+16, Height-y-34, 0.0f);
    glVertex3f(x - sl/2.-16, Height-y-34, 0.0f);
    glEnd();  

    // Draw nametag text
    if( !strcmp(n->o,"root") )    glColor3f(1.0f, 0.0f, 0.0f);
    else if( !strcmp(n->o,User) ) glColor3f(0.0f, 1.0f, 0.0f);
    else                          glColor3f(1.0f, 0.8f, 0.0f);
    glRasterPos2f(x - sl/2., Height-y-20);
    printGLf(&glw->font, "%s", n->n);
    if( (abs(x-Mx) < (sl/2.+16)) && (abs((Height-y)-My) < 32) ) glColor3f(1.0f, 1.0f, 0.0f);
    else                                                        glColor3f(1.0f, 1.0f, 1.0f);
    glRasterPos2f(x - sl/2., Height-y-4);
    printGLf(&glw->font, "pid: %d", n->pid);
    glRasterPos2f(x - sl/2., Height-y+12);
    printGLf(&glw->font, "owner: %s", n->o);
    glRasterPos2f(x - sl/2., Height-y+28);
    printGLf(&glw->font, "children: %d", n->nc);
    glLoadName(-1);

    // Draw a center point
    glColor3f(0.0f, 1.0f, 1.0f);
    glBegin(GL_QUADS);
    glVertex3f(x + -1, Height-y+1, 0.0f);
    glVertex3f(x +  1, Height-y+1, 0.0f);
    glVertex3f(x +  1, Height-y-1, 0.0f);
    glVertex3f(x + -1, Height-y-1, 0.0f);
    glEnd();  
    
    // Restore back to 3d with depth
    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glEnable(GL_DEPTH_TEST);
  } 

  // Let our children draw themselves
  for(i=0; i<n->nc; i++)
    DrawBoxesRecursive(glw, n->c[i]);
}

static void DrawTree(GLWindow *glw)
{ 
  // Push
  glPushMatrix();

  // Translate and rotate to set base matricies
  glTranslatef(Trans[0]/5.0f, -Trans[1]/5.0f, Trans[2]);
  glRotatef(Rot[0]/4.0f, 0.0f, 1.0f, 0.0f);
  glRotatef(Rot[1]/4.0f, 1.0f, 0.0f, 0.0f);

  // Save base matricies
  glGetDoublev(GL_PROJECTION_MATRIX, BaseProjection);
  glGetDoublev(GL_MODELVIEW_MATRIX, BaseModelView); 
  glGetIntegerv(GL_VIEWPORT, BaseViewport);

  // Call the recursive drawing function to display the tree
  DrawTreeRecursive(glw, Root);

  // Call the recursive drawing function to display the tree
  DrawBoxesRecursive(glw, Root);

  // Pop
  glPopMatrix();
}

/***************************************************************************************
 * HUD User Interface and helpers
 ***************************************************************************************/

static void DrawButton(GLWindow *glw, int x, int y, char *l)
{
  // Push
  glPushMatrix();
  glTranslatef(x, y, 0);

  // Draw the button border
  if( (abs(x-Mx) < 64) && (abs(y-(My-(Height-64))) < 12) ) glColor3f(1.0f, 1.0f, 0.0f);
  else                                                     glColor3f(1.0f, 1.0f, 1.0f);
  glBegin(GL_LINE_LOOP);
  glVertex2f(-64, -12);
  glVertex2f(-64,  12);
  glVertex2f( 64,  12);
  glVertex2f( 64, -12); 
  glEnd();  

  // Draw the label ( and Pop() )
  glPopMatrix();
  glRasterPos2f(x-(strlen(l)*glw->font.w)/2., y+glw->font.h/3.);
  printGLf(&glw->font, "%s", l);
}

// Draws the HUD control area
static void DrawHUD(GLWindow *glw)
{
  // Push
  glPushMatrix();

  // Switch to 2D with no depth testing
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(0., Width, Height, 0., 1, -100);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glDisable(GL_DEPTH_TEST);
  glTranslatef(0, Height-64, 0);

  // Draw HUD background and border
  glColor3f(0.0f, 1.0f, 1.0f);
  glColor4f(0.0f, 0.0f, 0.0f, 0.6f);
  glBegin(GL_QUADS);
  glVertex2f(0,     0);
  glVertex2f(0,     64);
  glVertex2f(Width, 64);
  glVertex2f(Width, 0);
  glEnd();
  glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
  glBegin(GL_LINES);
  glVertex2f(0,     0);
  glVertex2f(Width, 0);
  glEnd();  

  // Draw controls
  DrawButton(glw, 100, 32, "Clear Selection");
  DrawButton(glw, 300, 32, "Clear Anchors");
     
  // Switch back to 3D and enable depth testing
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glEnable(GL_DEPTH_TEST);
  
  // Pop
  glPopMatrix();
}

/***************************************************************************************
 * DrawScene()  ( Component Layout )
 ***************************************************************************************/

static void DrawScene(GLWindow *glw)
{
  // Clear the old scene
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glLoadIdentity();

  // Draw the tree
  DrawTree(glw);

  // Draw the HUD
  DrawHUD(glw);

  // Swap to display everything
  glXSwapBuffers(glw->dpy, glw->win);
}

/***************************************************************************************
 * Selection code
 ***************************************************************************************/

static pTreen GetNodeFromPoint(GLWindow *glw, int x, int y)
{
  GLuint sbuf[1024],*bufp=sbuf;
  GLuint hits,z1,z2;
  GLint  viewport[4];
  pTreen n;
  int    i,j;
  
  // Change to select mode and init names
  glSelectBuffer(1024, sbuf);
  glRenderMode(GL_SELECT);
  glInitNames();
  glPushName(0);
  
  // Set projection and model-view matricies
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glGetIntegerv(GL_VIEWPORT, viewport);
  gluPickMatrix((double)x, (double)(viewport[3]-y), 64, 64, viewport);
  gluPerspective(55.0f,(GLfloat)Width/(GLfloat)Height,0.1f,2000.0f);
  glTranslatef(0.0f, 0.0f, -80.0f);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  
  // Draw the scene (this has embeded names), flush, and return to render mode
  DrawTree(glw);
  glFlush(); 
  hits = glRenderMode(GL_RENDER);

  // Restore projection matrix
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  // Examin the hit buffer
  for(i=0; i<hits; i++) {
    j  = *(bufp++);
    z1 = *(bufp++);
    z2 = *(bufp++);
    while( j-- ) {
      // Really, just return the first match
      if( (n=SearchTree(Root, *(bufp++))) )
	return n;
    }
  }
  
  return NULL;
}

/***************************************************************************************
 * Ati-Gravity code, ruberband code, and vector ops
 ***************************************************************************************/

// d = a - b;
static void Vsub(double *a, double *b, double *d)
{
  int i;

  for(i=0; i<3; i++) d[i] = a[i] - b[i];
}

// d = a + b;
static void Vadd(double *a, double *b, double *d)
{
  int i;

  for(i=0; i<3; i++) d[i] = a[i] + b[i];
}

// a *= s;
static void Vscl(double *a, double s)
{
  int i;

  for(i=0; i<3; i++) a[i] *= s;
}

// Returns the distance from a to b 
static double Vdist(double *a, double *b)
{
  return sqrt( (b[0]-a[0])*(b[0]-a[0]) + 
	       (b[1]-a[1])*(b[1]-a[1]) + 
	       (b[2]-a[2])*(b[2]-a[2])   );
}

// a = normalize(a);
static void Vnrml(double *a)
{
  int i;  double d,o[3]={0.0,0.0,0.0};

  d = Vdist(o,a);
  for(i=0; i<3; i++) 
    a[i] /= d;
}

//**************************************************************************************

// Inner for loop for anti-grav velocity computation
static void CmpGVelRecur(pTreen c, pTreen n)
{
  int i;  double v[3],d;

  // We don't want to do anti-grav against ourselves
  if( c != n ) {
    // Get a normalized vector from n to c
    Vsub(c->l, n->l, v);
    Vnrml(v);
    // Contribute to velocity
    d = Vdist(c->l, n->l);
    Vscl(v, (1.0/(d*d))*GRAV_STRENGTH*FORCE_SCALE);
    Vadd(c->v, v, c->v);
  }

  // Let our children add their contribution
  for(i=0; i<n->nc; i++)
    CmpGVelRecur(c, n->c[i]);
}

// Outer for loop for anti-grav velocity computation
static void TreeCmpGVelRecur(pTreen n)
{
  int i;
  
  // Compute our velocity
  memset(n->v, 0, 3*sizeof(double));
  CmpGVelRecur(n, Root);
  
  // Compute our children's velocity
  for(i=0; i<n->nc; i++)
    TreeCmpGVelRecur(n->c[i]);
}

//**************************************************************************************

// Rubber-band velocity computation
static void TreeCmpRVelRecur(pTreen n)
{
  int i;  double v[3],d;
  
  // Compute our rubber band forces against our parent
  if( n->p ) {
    // Get a normalized vector from p to n
    Vsub(n->p->l, n->l, v);
    Vnrml(v);
    // Contribute to velocity
    d = Vdist(n->p->l, n->l);
    Vscl(v, d*BAND_STRENGTH*FORCE_SCALE);
    Vadd(n->v, v, n->v);
  }

  // Compute our rubber band forces against our children
  for(i=0; i<n->nc; i++) {
    // Get a normalized vector from n->c[i] to n
    Vsub(n->c[i]->l, n->l, v);
    Vnrml(v);
    // Contribute to velocity
    d = Vdist(n->l, n->c[i]->l);
    Vscl(v, d*BAND_STRENGTH*FORCE_SCALE);
    Vadd(n->v, v, n->v);
  }

  // Compute our children's velocity
  for(i=0; i<n->nc; i++)
    TreeCmpRVelRecur(n->c[i]);
}

//**************************************************************************************

// Initializes node loataions to something reasonable
static void TreeInitLocRecur(int l, int s, int w, pTreen n)
{
  int i;  double p[3] = {0.0,0.0,0.0};

  // Set our position and mass
  if( n->p )
    memcpy(p, n->p->l, 3*sizeof(double));
  n->l[0] = p[0] + (rand()%100) - 50;
  n->l[1] = p[1] + (rand()%100) - 50;
  n->l[2] = p[2] + (rand()%100) - 50;
  n->m = 1.0;

  /*
  n->l[0] = p[0] + cos(w*2*M_PI/(s+1))*15*(s/6.0f);
  n->l[1] = p[1] + -l*5.0f*(s/12.0f);
  n->l[2] = p[2] + sin(w*2*M_PI/(s+1))*15*(s/6.0f);
  */

  // Let our children set their position
  for(i=0; i<n->nc; i++)
    TreeInitLocRecur(l+1, n->nc, i, n->c[i]);
}

//**************************************************************************************

// Applies nodes' velocities to their locations
static void TreeAplyVelRecur(pTreen n)
{
  int i;
  
  // Apply velocity
  Vscl(n->v, n->m);
  Vadd(n->l, n->v, n->l);
  
  // Apply vel to our children
  for(i=0; i<n->nc; i++)
    TreeAplyVelRecur(n->c[i]);
}

/***************************************************************************************
 * Tree builders and helpers
 ***************************************************************************************/

void BuildFileTree(char *cd, pTreen n)
{
  struct dirent  *de;
  struct stat    ss;
  DIR            *d;
  char           buf[1024];
  pTreen         c;

  // Create a root node if needed
  if( !n ) {
    n = Root = (pTreen) malloc(sizeof(Treen));
    memset(Root, 0, sizeof(Treen));
    Root->o = "root";
    Root->n = cd;
    Root->pid = 0;
  }

  // Open
  if( !(d=opendir(cd)) )
    return;
  
  // Recursive Search
  while( (de=readdir(d)) ) {
    if( !strcmp(".",de->d_name) || !strcmp("..",de->d_name) ) continue;
    sprintf(buf,"%s/%s",cd,de->d_name);
    if(stat(buf, &ss))                                        continue;
    if(S_ISDIR(ss.st_mode)) {
      // Add directory child, name is de_dname
      n->c = realloc(n->c, (n->nc+1)*sizeof(pTreen));
      c = n->c[n->nc] = (pTreen) malloc(sizeof(Treen));
      memset(n->c[n->nc], 0, sizeof(Treen));
      n->c[n->nc]->p = n;
      c->d           = n->d+1;
      c->o           = "root"; 
      c->pid         = de->d_ino;
      c->n           = strdup(de->d_name); 
      n->nc++;
      // Check fir max pid and max depth (for color scaling)
      if(c->pid > MaxPid)   MaxPid   = c->pid;
      if(c->d   > MaxDepth) MaxDepth = c->d;
      BuildFileTree(buf,c);
      continue; 
    }
    // Add file child, name is de_dname
    n->c = realloc(n->c, (n->nc+1)*sizeof(pTreen));
    c = n->c[n->nc] = (pTreen) malloc(sizeof(Treen));
    memset(n->c[n->nc], 0, sizeof(Treen));
    n->c[n->nc]->p = n;
    c->d           = n->d+1;
    c->o           = "none"; 
    c->pid         = de->d_ino;
    c->n           = strdup(de->d_name); 
    n->nc++;
    // Check fir max pid and max depth (for color scaling)
    if(c->pid > MaxPid)   MaxPid   = c->pid;
    if(c->d   > MaxDepth) MaxDepth = c->d;
  }

  // Close
  closedir(d);

  // Final init
  if( n == Root )
    TreeInitLocRecur(0, 0, 0, Root);
}

static pTreen SearchTree(pTreen n, int pid)
{
  int i;  pTreen t;
  
  // Check for match
  if( n->pid == pid )
    return n;
  
  // Let children check for matches as well
  for(i=0; i<n->nc; i++)
    if( (t=SearchTree(n->c[i], pid)) )
      return t;

  // No match found
  return NULL;
}

void BuildProcessTree(char *fn)
{
  FILE *ps;  char line[1024], trash[512], owner[128], *cmd;  pTreen p,c;  int i,pid,ppid;

  // Open input for parsing
  if(fn) {
    if( !(ps=fopen(fn, "r")) ) {
      perror("Main(): StartTreeView(): BuildProcessTree()");
      exit(1);
    }
  } else {
    // `ps` as a pipe is the default
    if( !(ps=popen("ps -ef", "r")) ) {
      perror("Main(): StartTreeView(): BuildProcessTree()");
      exit(1);
    }
  }

  // Create a root node
  Root = (pTreen) malloc(sizeof(Treen));
  memset(Root, 0, sizeof(Treen));
  fgets(line, sizeof(line), ps);
  if( !fgets(line, sizeof(line), ps) ) {
    fprintf(stderr, "`ps` output has too few lines.\n");
    exit(1);
  }
  // Scan init line
  if( sscanf(line, "%s %d %d %s %s %s %s %n", owner, &pid, &ppid, trash, trash, trash, trash, &i) != 7 ) {
    fprintf(stderr, "`ps` init line is malformed.\n");
    exit(1);
  }
  Root->o   = strdup(owner);
  Root->pid = pid;
  Root->n   = strdup(line+i);

  // Parse the ps output one line at a time
  fgets(line, sizeof(line), ps);
  while( fgets(line,sizeof(line),ps) ) {
    if( !strlen(line) ) {
      fprintf(stderr, "`ps` line is too short.\n");
      break;
    }
    line[strlen(line)-1] = 0;

    // Scan PS line
    if( sscanf(line, "%s %d %d %s %s %s %s %n", owner, &pid, &ppid, trash, trash, trash, trash, &i) != 7 ) {
      fprintf(stderr, "`ps` line is malformed.\n");
      break;
    }
    cmd = strdup(line+i);

    // Find parent node
    if( !(p=SearchTree(Root, ppid)) ) {
      fprintf(stderr, "No parent found for ppid: %d\n", ppid);
      continue;
    }
    
    // Install child in parent
    p->c = realloc(p->c, (p->nc+1)*sizeof(pTreen));
    c = p->c[p->nc] = (pTreen) malloc(sizeof(Treen));
    memset(p->c[p->nc], 0, sizeof(Treen));
    p->c[p->nc]->p = p;
    c->d           = p->d+1;
    c->o           = strdup(owner);
    c->pid         = pid;
    c->n           = cmd;
    p->nc++;

    // Check fir max pid and max depth (for color scaling)
    if(pid  > MaxPid)   MaxPid   = pid;
    if(c->d > MaxDepth) MaxDepth = c->d;
  }

  // Close ps
  if(fn) fclose(ps);
  else   pclose(ps);

  // Init the node positions
  TreeInitLocRecur(0, 0, 0, Root);
}

/***************************************************************************************
 * Window creation and event handler and helpers
 ***************************************************************************************/

void *DynamicsThread(void *arg)
{
  // Just push the dynamics forever
  while(1) {
    // Lock
    pthread_mutex_lock(&TreeMutex);

    // Update positions based on anti-grav
    TreeCmpGVelRecur(Root);
    TreeCmpRVelRecur(Root);
    TreeAplyVelRecur(Root);

    // Unlock
    pthread_mutex_unlock(&TreeMutex);

    // Yeild to be nice
    yeild(10000);
  }

  // Just to shut up the compiler
  return NULL;
}

static void StartDynamicsThread()
{
  pthread_t t;  pthread_attr_t a;
  
  // Create a mutex for thread sync
  pthread_mutex_init(&TreeMutex,NULL);
  
  // Set up the thread attributes
  pthread_attr_init(&a);
  pthread_attr_setdetachstate(&a,PTHREAD_CREATE_DETACHED);
  pthread_attr_setscope(&a,PTHREAD_SCOPE_SYSTEM);

  // Create dynamics thread
  if( pthread_create(&t, &a, DynamicsThread, NULL) ) {
    perror("main(): StartTreeView(): StartDynamicsThread(): pthread_create()");
    exit(1);
  }
}

static void ClearAnchorRecur(pTreen n)
{
  int i;

  // Clear our anchor flag
  n->m = 1;

  // Let our children clear their flags
  for(i=0; i<n->nc; i++)
    ClearAnchorRecur(n->c[i]);
}

static void ClearSelectionRecur(pTreen n)
{
  int i;

  // Clear our selection 
  n->s = 0;

  // Let our children clear their flags
  for(i=0; i<n->nc; i++)
    ClearSelectionRecur(n->c[i]);
}

static void StartTreeView(char *fn)
{
  XEvent event;  GLWindow glw;  KeySym key;  char keys[255],d=0;
  int    done=0,x=0,y=0;
  double px,py,pz;  pTreen n;

  // Set the User global
  if(getenv("USER")) sprintf(User, "%s", getenv("USER"));
  else               sprintf(User, "none");

  // Create the window and initialize it
  glw.id = 0;
  CreateGLWindow(&glw, "pTree", Width, Height);
  InitGLWindow(&glw);

  // Build  tree (alternate:  BuildFileTree(fn, Root);)
  BuildProcessTree(fn);
  //BuildFileTree(fn, Root);

  // Start the dynamics thread
  StartDynamicsThread();

  // Give a decent start position
  Trans[0] =  0;
  Trans[1] = -150;
  Trans[2] = -480;

  // Process events until we are "done"
  while (!done){
    
    // Is there an available event?
    if (XCheckIfEvent(glw.dpy, &event, AnyEvent, NULL)){
      switch (event.type) {
      case Expose:
        // Redraw
        if (event.xexpose.count != 0)
          break;
        DrawScene(&glw);
        break;
      case ConfigureNotify:
        // Resize
        if ((event.xconfigure.width != glw.width) || (event.xconfigure.height != glw.height)) {
          Width  = glw.width = event.xconfigure.width;
          Height = glw.height = event.xconfigure.height;
          ResizeGLScene(event.xconfigure.width, event.xconfigure.height);
        }
        break;
      case ClientMessage:    
        // Close button
        if (*XGetAtomName(glw.dpy, event.xclient.message_type) == *"WM_PROTOCOLS")
          done = 1;
        break;
      case KeyPress:
        // Key press
        if( XLookupString(&event.xkey, keys, 255, &key, 0) == 1 ) {
          switch (*keys){
          case '+': 
            break;
          case '-':
            break;
          }
        }
        break;
      case ButtonRelease:
        // Mouse up
        d = 0;
        break;
      case MotionNotify:
        // Mouse move
        Mx = event.xbutton.x;
        My = event.xbutton.y;

        switch(d) {
	case 0:
	  Hover = GetNodeFromPoint(&glw, Mx, My);
	  break;
        case 1:
          Trans[0] += event.xbutton.x-x;
          Trans[1] += event.xbutton.y-y;
          break;
        case 2:
	  // Drag node around in space
	  if( !Selected )
	    break;
	  gluProject(Selected->l[0], Selected->l[1], Selected->l[2], BaseModelView, BaseProjection, BaseViewport, &px, &py, &pz);
	  px += event.xbutton.x-x;
	  py -= event.xbutton.y-y;
	  gluUnProject(px, py, pz, BaseModelView, BaseProjection, BaseViewport, &Selected->l[0], &Selected->l[1], &Selected->l[2]);
	  break;
        case 3:
          Rot[0] += event.xbutton.x-x;
          Rot[1] += event.xbutton.y-y;
	  break;
        }
	x = event.xbutton.x;
	y = event.xbutton.y;
        break;
      case ButtonPress:
        // Mouse down
        switch (event.xbutton.button){
        case 1: 
	  // Check for command area
	  if( event.xbutton.y > Height-64 ) {
	    // Check for "Clear Selection" press
	    if( (abs(100-event.xbutton.x) < 64) && (abs((Height-64+32)-event.xbutton.y) < 12) )
	      ClearSelectionRecur(Root);
	    // Check for "Clear Anchor" press
	    if( (abs(300-event.xbutton.x) < 64) && (abs((Height-64+32)-event.xbutton.y) < 12) )
	      ClearAnchorRecur(Root);
	    break;
	  }
          x = event.xbutton.x; 
          y = event.xbutton.y;
	  // Toggle selected
	  if( (n = GetNodeFromPoint(&glw, x, y)) )
	    n->s ^= 1;
          d = 1;
          break;
        case 2:
	  // Check for command area
	  if( event.xbutton.y > Height-64 ) break;
	  x = event.xbutton.x; 
          y = event.xbutton.y; 
	  // Toggle anchor
	  if( (Selected=GetNodeFromPoint(&glw, x, y)) )
	    Selected->m = ((Selected->m)?(0.0):(1.0));
          d = 2;
	  break;
        case 3:
	  // Check for command area
	  if( event.xbutton.y > Height-64 ) break;
	  x = event.xbutton.x; 
          y = event.xbutton.y;
	  d = 3;
	  break;
        case 4:
	  Trans[2] -= 10;
	  break;
        case 5:
	  Trans[2] += 10;
	  break;
        }
      }
    } else {
      // Push frames
      pthread_mutex_lock(&TreeMutex);
      DrawScene(&glw);
      pthread_mutex_unlock(&TreeMutex);

      // Yeild to be nice
      yeild(50000);
    }
  }
  
  /* We are done, kill the window */
  KillGLWindow(&glw);
}

/***************************************************************************************
 * Application entry point
 ***************************************************************************************/

int main(int argc, char **argv)
{
  // Start the display
  StartTreeView(argv[1]);

  // Return success
  return 0;
}
