#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <math.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/select.h>
#include <pthread.h>
#include <unistd.h>
#include <GL/glx.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#include "mchat.h"

// Maps "virtual" socket calls to the real socket interface
#if 1
#define Socket(a,b,c)          socket(a,b,c)
#define Bind(a,b,c)            bind(a,b,c)
#define Sendto(a,b,c,d,e,f)    sendto(a,b,c,d,e,f)
#define Recvfrom(a,b,c,d,e,f)  recvfrom(a,b,c,d,e,f)
#endif

/***************************************************************************************
 * Gobals
 ***************************************************************************************/

// Width and Height of the window and mouse coordinates respectivly
int     Width = 800,   Height = 600;   
int     Mx    = 10000, My     = 10000;
int     Vmx   = 10000, Vmy    = 10000;

// Speed the player should move at
float   KeySpeed = 12.0f;

// Player position
point3f Pos;

// Lighting information
GLfloat LightAmbient[]  = { 0.5f, 0.5f, 0.5f, 1.0f };
GLfloat LightDiffuse[]  = { 1.0f, 1.0f, 1.0f, 1.0f };
GLfloat LightPosition[] = { 0.0f, 0.0f, 2.0f, 1.0f };	

// Display lists for some of the various objects we need to draw
int     CLSkybox;
int     CLWorld;
int     CLAvatar;
int     CLWater;
int     Avatars[7];
int     nAvatars=7;
int     Avatar;

// Data structures used for thread sync
pthread_mutex_t Mutex;
char            ClientSockBound;

// Network data
int                Mode;
struct sockaddr_in SrvAddr;
int                AddrLen;
int                Sock;
unsigned long long nPacketsIn;
unsigned long long nPacketsOut;
unsigned long long nBytesIn;
unsigned long long nBytesOut;

// Chat state info
Participant Participants[MAX_PARTICIPANTS];
int         nParticipants;
char        Name[8];
char        Text[32];
int         WaterLevel = 75;

/***************************************************************************************
 * These are lock/unlock wrappers to make this whole mess thread safe
 ***************************************************************************************/

void LockMutex()
{
  if( pthread_mutex_lock(&Mutex) ) {
    perror("LockMutex(): pthread_mutex_lock()");
    exit(1);
  }
}

void UnlockMutex()
{
  if( pthread_mutex_unlock(&Mutex) ) {
    perror("UnlockMutex(): pthread_mutex_unlock()");
    exit(1);
  }
}


/***************************************************************************************
 * Font / GL / X / etc
 ***************************************************************************************/

static GLuint BuildFont(GLWindow *glw, char *fname)
{
  XFontStruct *font;  GLuint fbase;   
    
  /* Storage for 96 characters */
  fbase = glGenLists(96);      

  /* Load a font with a specific name in "Host Portable Character Encoding" */
  if ( !(font = XLoadQueryFont(glw->dpy, fname)) ) {
    fprintf(stderr,"BuildFont: Could not load font \"%s\"\n",fname);
    exit(-1);
  }

  /* Build 96 display lists out of our font starting at char 32 */
  glXUseXFont(font->fid, 32, 96, fbase);
  XFreeFont(glw->dpy, font);
  
  return fbase;
}

static void DeleteFont(GLuint font)
{
  glDeleteLists(font, 96);
}

// Like printf, but for OpenGL
static void printGLf(GLuint font, const char *fmt, ...)
{
  va_list ap;  char text[256];

  if (fmt == NULL) return;

  va_start(ap, fmt);  
  vsprintf(text, fmt, ap);
  va_end(ap);
  glPushAttrib(GL_LIST_BIT);
  glListBase(font - 32);
  glCallLists(strlen(text), GL_UNSIGNED_BYTE, text);
  glPopAttrib();
}

// Loads 24-bit bitmaps
void LoadBitmap(char *fn, textureImage *tex)
{
  FILE               *file;
  unsigned short int  bfType;
  int                 bfOffBits;
  short int           biPlanes, biBitCount;
  int                 biSizeImage, i;
  unsigned char       t;
  
  // Open the file 
  if ((file = fopen(fn, "rb")) == NULL) {
    fprintf(stderr, "LoadBitmap(): Could not open file: %s\n", fn);
    exit(1);
  }
  // Check for bitmap marker
  if(!fread(&bfType, sizeof(short int), 1, file)) {
    fprintf(stderr, "LoadBitmap(): Error reading file.\n");
    exit(1);
  }
  if (bfType != 19778) {
    fprintf(stderr, "LoadBitmap(): Not a Bitmap-File.\n");
    exit(1);
  }        
  // Skip past some parts of the header we don't care about
  fseek(file, 8, SEEK_CUR);
  // Read the position of the actual bitmap data
  if (!fread(&bfOffBits, sizeof(int), 1, file)) {
    fprintf(stderr, "LoadBitmap(): Error reading file.\n");
    exit(1);
  }
  // Skip more header we don't care about
  fseek(file, 4, SEEK_CUR);
  // Read the width of the bitmap
  if( !fread(&tex->width, sizeof(int), 1, file) ) {
    fprintf(stderr, "LoadBitmap(): Error reading file.\n");
    exit(1);
  }
  // Read the height of the bitmap
  if( !fread(&tex->height, sizeof(int), 1, file) ) {
    fprintf(stderr, "LoadBitmap(): Error reading file.\n");
    exit(1);
  }
  // Read the number of planes (though I'm not sure that these are...)
  fread(&biPlanes, sizeof(short int), 1, file);
  if (biPlanes != 1) {
    fprintf(stderr, "LoadBitmap(): Error: number of planes must be 1.\n");
    exit(1);
  }
  // Read the bpp
  if (!fread(&biBitCount, sizeof(short int), 1, file)) {
    fprintf(stderr, "LoadBitmap(): Error reading file.\n");
    exit(1);
  }
  if (biBitCount != 24) {
    fprintf(stderr, "LoadBitmap(): Bits per Pixel not 24.\n");
    exit(1);
  }
  // Calculate the size of the image in bytes
  biSizeImage = tex->width * tex->height * 3;
  // Allocate space for the bitmap in memory
  if( !(tex->data = malloc(biSizeImage)) ) {
    fprintf(stderr, "LoadBitmap(): Error allocating bitmap memory.\n");
    exit(1);
  }
  // Seek to the actual data
  fseek(file, bfOffBits, SEEK_SET);
  if (!fread(tex->data, biSizeImage, 1, file)) {
    fprintf(stderr, "LoadBitmap(): Error loading file.\n");
    exit(1);
  }
  // Swap red and blue (bgr -> rgb)
  for (i = 0; i < biSizeImage; i += 3) {
    t = tex->data[i];
    tex->data[i] = tex->data[i+2];
    tex->data[i+2] = t;
  } 
  // Close the file
  fclose(file);
}

int LoadTexture(char *fn)
{
  textureImage bmp;
  int          tex;
  
  // Actually read in the bitmap file
  LoadBitmap(fn, &bmp);
  
  // Turn that into an OpenGL texture
  glGenTextures(1, &tex);
  glBindTexture(GL_TEXTURE_2D, tex);
  glTexImage2D(GL_TEXTURE_2D, 0, 3, bmp.width, bmp.height, 0,
	       GL_RGB, GL_UNSIGNED_BYTE, bmp.data);
  
  // Use linear filtering
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  
  // Free the raw texture data, we don't need it anymore
  free(bmp.data);
  
  // Return the new texture we just built
  return tex;
}

// Called when window is resized
static void ResizeGLScene(unsigned int width, unsigned int height)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glViewport(0, 0, width, height); 
  gluPerspective(55.0f,(GLfloat)width/(GLfloat)height,1.0f,20000.0f);
  glTranslatef(0.0f, 0.0f, -500.0f);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
}

// Kills an OpenGL window
static void KillGLWindow(GLWindow *glw)
{
  if (glw->ctx) {
    glDeleteLists(0, 65535);
    if (!glXMakeCurrent(glw->dpy, None, NULL))
      fprintf(stderr, "KillGLWindow: Could not release drawing context.\n");
    glXDestroyContext(glw->dpy, glw->ctx);
    glw->ctx = NULL;
  }
  XCloseDisplay(glw->dpy);
}

// Creates an OpenGL enabled window
static void CreateGLWindow(GLWindow *glw, char* title, int width, int height)
{
  XVisualInfo *vi;  Colormap cmap;  Window twin;  Atom wmDelete;  int glxM, glxm, t;
    
  // Connect to X
  if( !(glw->dpy    = XOpenDisplay(0)) ) {
    fprintf(stderr,"Cannot connect to X server!\n");
    exit(1);
  }
  glw->screen = DefaultScreen(glw->dpy);

  // Get the appropriate visual
  glXQueryVersion(glw->dpy, &glxM, &glxm);
  printf("%d: glX-Version %d.%d\n", glw->id, glxM, glxm);
  if ( (vi = glXChooseVisual(glw->dpy, glw->screen, attrListDbl)) ) {
    printf("%d: Selected doublebuffered mode.\n", glw->id);
  } else {
    vi = glXChooseVisual(glw->dpy, glw->screen, attrListSgl);
    printf("%d: Selected singlebuffered mode.\n", glw->id);
  }

  // Create a GLX context and color map
  glw->ctx               = glXCreateContext(glw->dpy, vi, 0, GL_TRUE);
  cmap                   = XCreateColormap(glw->dpy, RootWindow(glw->dpy, vi->screen), vi->visual, AllocNone);
  glw->attr.colormap     = cmap;
  glw->attr.border_pixel = 0;

  // Create the window
  glw->attr.event_mask = ExposureMask | KeyPressMask | ButtonPressMask| ButtonReleaseMask | StructureNotifyMask | PointerMotionMask;
  glw->win = XCreateWindow(glw->dpy, RootWindow(glw->dpy, vi->screen),
                           0, 0, width, height, 0, vi->depth, InputOutput, vi->visual,
                           CWBorderPixel | CWColormap | CWEventMask, &glw->attr);
  wmDelete = XInternAtom(glw->dpy, "WM_DELETE_WINDOW", True);
  XSetWMProtocols(glw->dpy, glw->win, &wmDelete, 1);
  XSetStandardProperties(glw->dpy, glw->win, title, title, None, NULL, 0, NULL);
  XMapRaised(glw->dpy, glw->win);
    
  // Connect the glx-context to the window
  glXMakeCurrent(glw->dpy, glw->win, glw->ctx);
  XGetGeometry(glw->dpy, glw->win, &twin, &glw->x, &glw->y,
               &glw->width, &glw->height, (unsigned int*)&t, &glw->depth);
  printf("%d: Depth %d\n", glw->id, glw->depth);
  if (glXIsDirect(glw->dpy, glw->ctx)) printf("%d: Direct Rendering enabled.\n", glw->id);
  else                                 printf("%d: Direct Rendering disabled.\n", glw->id);
}

void Yeild(int usec)
{
  struct timeval tv;

  // Sleep for a few ms
  tv.tv_sec  = 0;
  tv.tv_usec = usec;
  select(0, NULL,  NULL, NULL, &tv);
}

static int AnyEvent(Display *d, XEvent *e, XPointer p)
{
  return True;
}

static void InitGLWindow(GLWindow *glw)
{  
  GLfloat fogColor[4]= {0.5f, 0.5f, 0.5f, 1.0f};
  
  // Basic stuff
  glShadeModel(GL_SMOOTH);
  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
  glClearDepth(1.0f);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  // Lighting
  glLightfv(GL_LIGHT1, GL_AMBIENT,  LightAmbient);
  glLightfv(GL_LIGHT1, GL_DIFFUSE,  LightDiffuse);
  glLightfv(GL_LIGHT1, GL_POSITION, LightPosition);
  // Texture
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_NEAREST);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR_MIPMAP_NEAREST);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
  glEnable(GL_TEXTURE_2D);
  // Fog
  glFogi(GL_FOG_MODE, GL_LINEAR );
  glFogfv(GL_FOG_COLOR, fogColor);
  glFogf(GL_FOG_DENSITY, 0.35f);
  glHint(GL_FOG_HINT, GL_DONT_CARE);
  glFogf(GL_FOG_START, 300.0f);
  glFogf(GL_FOG_END, 1000.0f);
  // Setup the default bitmap font
  glw->font = BuildFont(glw, "fixed");   
  // Set up the perspective
  ResizeGLScene(glw->width, glw->height);
  // Flush
  glFlush();
}

void glOutlineCircle(int x, int y, double r)
{
  int k;

  glBegin(GL_LINE_LOOP);
  for(k=0;k<32;k++)
    glVertex2d(x+r*cos(k*2*PI/32),y+r*sin(k*2*PI/32));
  glEnd();
}

void glCircle(int x, int y, double r)
{
  int k;

  glBegin(GL_POLYGON);
  for(k=0;k<32;k++)
    glVertex2d(x+r*cos(k*2*PI/32),y+r*sin(k*2*PI/32));
  glEnd();
}

/***************************************************************************************
 * Functions which build OpenGL call lists
 ***************************************************************************************/

void BuildSkyBox(GLWindow *glw)
{
  int   stup,stdown,stleft,stright,stback,stfront;
  float a,b;

  // I use these values to slightly tweak the skybox textures
  a = 8/4500.0f;
  b = 1.0f-8/4500.0f;

  // I don't really see anything wrong with loading textures
  // here.  There are not many of them, and this whole section
  // is done only once anyways.
  stup    = LoadTexture("data/SkyUp.bmp"   ); 
  stdown  = LoadTexture("data/SkyDown.bmp" ); 
  stleft  = LoadTexture("data/SkyLeft.bmp" ); 
  stright = LoadTexture("data/SkyRight.bmp"); 
  stback  = LoadTexture("data/SkyBack.bmp" ); 
  stfront = LoadTexture("data/SkyFront.bmp");

  glNewList(CLSkybox=glGenLists(1),GL_COMPILE);
  glColor3f(1.0f,1.0f,1.0f);
  glBindTexture(GL_TEXTURE_2D,stup);
  glBegin(GL_QUADS);
  glTexCoord2f(b,b); glVertex3i(-10000, 10000,  10000);
  glTexCoord2f(a,b); glVertex3i(-10000, 10000, -10000);
  glTexCoord2f(a,a); glVertex3i( 10000, 10000, -10000);
  glTexCoord2f(b,a); glVertex3i( 10000, 10000,  10000);
  glEnd();
  glBindTexture(GL_TEXTURE_2D,stdown);
  glBegin(GL_QUADS);
  glTexCoord2f(b,a); glVertex3i(-10000,-10000,  10000);
  glTexCoord2f(b,b); glVertex3i( 10000,-10000,  10000);
  glTexCoord2f(a,b); glVertex3i( 10000,-10000, -10000);
  glTexCoord2f(a,a); glVertex3i(-10000,-10000, -10000);
  glEnd();
  glBindTexture(GL_TEXTURE_2D,stleft);
  glBegin(GL_QUADS);
  glTexCoord2f(a,a); glVertex3i(-10000,-10000, 10000);
  glTexCoord2f(b,a); glVertex3i(-10000,-10000,-10000);
  glTexCoord2f(b,b); glVertex3i(-10000, 10000,-10000);
  glTexCoord2f(a,b); glVertex3i(-10000, 10000, 10000);
  glEnd();
  glBindTexture(GL_TEXTURE_2D,stright);
  glBegin(GL_QUADS);
  glTexCoord2f(b,a); glVertex3i( 10000,-10000, 10000);
  glTexCoord2f(b,b); glVertex3i( 10000, 10000, 10000);
  glTexCoord2f(a,b); glVertex3i( 10000, 10000,-10000);
  glTexCoord2f(a,a); glVertex3i( 10000,-10000,-10000);
  glEnd();
  glBindTexture(GL_TEXTURE_2D,stback);
  glBegin(GL_QUADS);
  glTexCoord2f(a,a); glVertex3i(-10000,-10000,-10000);
  glTexCoord2f(b,a); glVertex3i( 10000,-10000,-10000);
  glTexCoord2f(b,b); glVertex3i( 10000, 10000,-10000);
  glTexCoord2f(a,b); glVertex3i(-10000, 10000,-10000);
  glEnd();
  glBindTexture(GL_TEXTURE_2D,stfront);
  glBegin(GL_QUADS);
  glTexCoord2f(b,a); glVertex3i(-10000,-10000, 10000);
  glTexCoord2f(b,b); glVertex3i(-10000, 10000, 10000);
  glTexCoord2f(a,b); glVertex3i( 10000, 10000, 10000);
  glTexCoord2f(a,a); glVertex3i( 10000,-10000, 10000);
  glEnd();
  glEndList();
}

void BuildWorld(GLWindow *glw)
{
  int wall,floor;

  // I don't really see anything wrong with loading textures
  // here.  There are not many of them, and this whole section
  // is done only once anyways.
  wall  = LoadTexture("data/wall.bmp"); 
  floor = LoadTexture("data/floor.bmp"); 

  // Draw the static world people are allowed to run around in.
  glNewList(CLWorld=glGenLists(1), GL_COMPILE);
  glColor3f(1.0f,1.0f,1.0f);
  glBindTexture(GL_TEXTURE_2D, wall);
  glBegin(GL_QUADS);
  // Front
  glNormal3f(0.0f, -1.0f, 0.0f);
  glTexCoord2f(6.0f,1.0f); glVertex3f(-500.0f, 0.0f,   -500.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f( 500.0f, 0.0f,   -500.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f( 500.0f, 100.0f, -500.0f);
  glTexCoord2f(6.0f,0.0f); glVertex3f(-500.0f, 100.0f, -500.0f);
  // Back
  glNormal3f(0.0f, 1.0f, 0.0f);
  glTexCoord2f(6.0f,1.0f); glVertex3f(-500.0f, 0.0f,    500.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f( 500.0f, 0.0f,    500.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f( 500.0f, 100.0f,  500.0f);
  glTexCoord2f(6.0f,0.0f); glVertex3f(-500.0f, 100.0f,  500.0f);
  // Right
  glNormal3f(1.0f, 0.0f, 0.0f);
  glTexCoord2f(6.0f,1.0f); glVertex3f( 500.0f, 0.0f,   -500.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f( 500.0f, 0.0f,    500.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f( 500.0f, 100.0f,  500.0f);
  glTexCoord2f(6.0f,0.0f); glVertex3f( 500.0f, 100.0f, -500.0f);
  // Left
  glNormal3f(-1.0f, 0.0f, 0.0f);
  glTexCoord2f(6.0f,1.0f); glVertex3f(-500.0f, 0.0f,   -500.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f(-500.0f, 0.0f,    500.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f(-500.0f, 100.0f,  500.0f);
  glTexCoord2f(6.0f,0.0f); glVertex3f(-500.0f, 100.0f, -500.0f);
  glEnd();  
  // Bottom
  glBindTexture(GL_TEXTURE_2D, floor);
  glBegin(GL_QUADS);
  glNormal3f(0.0f, 1.0f, 0.0f);
  glTexCoord2f(6.0f,6.0f); glVertex3f(-500.0f, 0.0f,   -500.0f);
  glTexCoord2f(0.0f,6.0f); glVertex3f(-500.0f, 0.0f,    500.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f( 500.0f, 0.0f,    500.0f);
  glTexCoord2f(6.0f,0.0f); glVertex3f( 500.0f, 0.0f,   -500.0f);
  glEnd();  
  // End the call list
  glEndList();
}

void BuildAvatar(GLWindow *glw)
{
  int i;  char buf[128];

  // I don't really see anything wrong with loading textures
  // here.  There are not many of them, and this whole section
  // is done only once anyways.
  for(i=0; i<nAvatars; i++) {
    sprintf(buf, "data/avatar%d.bmp", i);
    Avatars[i] = LoadTexture(buf); 
  }

  // Draw the static world people are allowed to run around in.
  glNewList(CLAvatar=glGenLists(1), GL_COMPILE);
  glColor3f(1.0f,1.0f,1.0f);
  glBegin(GL_QUADS);
  // Front
  glNormal3f(0.0f, -1.0f, 0.0f);
  glTexCoord2f(1.0f,1.0f); glVertex3f(-25.0f, -25.0f, -25.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f( 25.0f, -25.0f, -25.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f( 25.0f,  25.0f, -25.0f);
  glTexCoord2f(1.0f,0.0f); glVertex3f(-25.0f,  25.0f, -25.0f);
  // Back
  glNormal3f(0.0f, 1.0f, 0.0f);
  glTexCoord2f(1.0f,1.0f); glVertex3f(-25.0f, -25.0f,  25.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f( 25.0f, -25.0f,  25.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f( 25.0f,  25.0f,  25.0f);
  glTexCoord2f(1.0f,0.0f); glVertex3f(-25.0f,  25.0f,  25.0f);
  // Right
  glNormal3f(1.0f, 0.0f, 0.0f);
  glTexCoord2f(1.0f,1.0f); glVertex3f( 25.0f, -25.0f, -25.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f( 25.0f, -25.0f,  25.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f( 25.0f,  25.0f,  25.0f);
  glTexCoord2f(1.0f,0.0f); glVertex3f( 25.0f,  25.0f, -25.0f);
  // Left
  glNormal3f(-1.0f, 0.0f, 0.0f);
  glTexCoord2f(1.0f,1.0f); glVertex3f(-25.0f, -25.0f, -25.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f(-25.0f, -25.0f,  25.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f(-25.0f,  25.0f,  25.0f);
  glTexCoord2f(1.0f,0.0f); glVertex3f(-25.0f,  25.0f, -25.0f);
  // Bottom
  glNormal3f(0.0f, 1.0f, 0.0f);
  glTexCoord2f(1.0f,1.0f); glVertex3f(-25.0f, -25.0f, -25.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f(-25.0f, -25.0f,  25.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f( 25.0f, -25.0f,  25.0f);
  glTexCoord2f(1.0f,0.0f); glVertex3f( 25.0f, -25.0f, -25.0f);
  // Top
  glNormal3f(0.0f, 1.0f, 0.0f);
  glTexCoord2f(1.0f,1.0f); glVertex3f(-25.0f,  25.0f, -25.0f);
  glTexCoord2f(0.0f,1.0f); glVertex3f(-25.0f,  25.0f,  25.0f);
  glTexCoord2f(0.0f,0.0f); glVertex3f( 25.0f,  25.0f,  25.0f);
  glTexCoord2f(1.0f,0.0f); glVertex3f( 25.0f,  25.0f, -25.0f);
  // End the call list
  glEnd();  
  glEndList();
}

void BuildWater(GLWindow *glw)
{
  int water;

  // I don't really see anything wrong with loading textures
  // here.  There are not many of them, and this whole section
  // is done only once anyways.
  water = LoadTexture("data/water.bmp"); 

  // Draw the water plane
  glNewList(CLWater=glGenLists(1), GL_COMPILE);
  glBindTexture(GL_TEXTURE_2D, water);
  glBegin(GL_QUADS);
  glNormal3f(0.0f, 1.0f, 0.0f);
  glTexCoord2f(8.0f, 8.0f); glVertex3f(-500.0f, 0.0f, -500.0f);
  glTexCoord2f(0.0f, 8.0f); glVertex3f(-500.0f, 0.0f,  500.0f);
  glTexCoord2f(0.0f, 0.0f); glVertex3f( 500.0f, 0.0f,  500.0f);
  glTexCoord2f(8.0f, 0.0f); glVertex3f( 500.0f, 0.0f, -500.0f);
  glEnd();  
  glEndList();
}

/***************************************************************************************
 * OpenGL drawing functions
 ***************************************************************************************/

void DrawParticipants(GLWindow *glw)
{
  int i;

  LockMutex();

  // Draw all participants
  for(i=0; i<nParticipants; i++) {
    // We don't want to draw ourselves
    if( !strcmp(Name,Participants[i].name) )
      continue;

    // Draw a standard avatar    
    glPushMatrix();
    glTranslatef(-Participants[i].pos.x, -Participants[i].pos.y, -Participants[i].pos.z);
    if( Participants[i].avatar > (nAvatars-1) )
      Participants[i].avatar = nAvatars-1;
    glBindTexture(GL_TEXTURE_2D, Avatars[Participants[i].avatar]);
    glCallList(CLAvatar);
    
    // Draw the name
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_DEPTH_TEST);
    glColor3f(0.0f, 1.0f, 1.0f);
    glRasterPos3f(0.0f, 0.0f, 0.0f);
    printGLf(glw->font, "%s", Participants[i].name);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_TEXTURE_2D);
    glPopMatrix();
  }

  UnlockMutex();

}

void DrawWater()
{
  LockMutex();
  
  glPushMatrix();
  glEnable(GL_ALPHA);
  glColor4f(0.0f, 0.0f, 1.0f, 0.7f);
  glTranslatef(0.0f, 75.0f, 0.0f); 
  glCallList(CLWater);
  glColor3f(1.0f, 1.0f, 1.0f);
  glDisable(GL_ALPHA);
  glPopMatrix();

  UnlockMutex();
}

void DrawHUD(GLWindow *glw)
{ 
  time_t t;

  LockMutex();

  // Save old projection matrix
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();

  // Build new 2d ortho projection matrix
  glLoadIdentity();
  glViewport(0, 0, Width, Height);  
  glOrtho(0., Width, Height, 0., 1, -100);

  // Save and reset model view matrix
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  // Now draw everything in 2D
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_TEXTURE_2D);
  // Draw translucent HUD area
  glEnable(GL_ALPHA);
  glColor4f(0.0f, 0.0f, 0.0f, 0.5f);
  glBegin(GL_QUADS);
  glVertex2i(0,     Height);
  glVertex2i(0,     Height-100);
  glVertex2i(Width, Height-100);
  glVertex2i(Width, Height);
  glEnd();  
  glDisable(GL_ALPHA);
  // Draw stats over the HUD area
  glColor3f(1.0f, 0.0f, 1.0f);
  glRasterPos2i(30, Height-47);
  printGLf(glw->font, "Participants: %d", nParticipants);
  glRasterPos2i(170, Height-47);
  printGLf(glw->font, "Packets: %lld/%lld (in/out)", nPacketsIn, nPacketsOut);
  glRasterPos2i(370, Height-47);
  printGLf(glw->font, "KB: %lld/%lld (in/out)", nBytesIn/1024, nBytesOut/1024);
  glRasterPos2i(520.0f, Height-47);
  t = time(NULL);
  printGLf(glw->font, "%s", ctime(&t));
  glEnable(GL_TEXTURE_2D);
  // Draw avatar selector square 
  glColor3f(1.0f, 1.0f, 1.0f);
  glTranslatef(Width-32-16, Height-32-16, 0); 
  glBindTexture(GL_TEXTURE_2D, Avatars[Avatar]);
  glBegin(GL_QUADS); 
  glTexCoord2f(0.0f, 0.0f); glVertex2i(-32,  32);
  glTexCoord2f(0.0f, 1.0f); glVertex2i(-32, -32);
  glTexCoord2f(1.0f, 1.0f); glVertex2i( 32, -32);
  glTexCoord2f(1.0f, 0.0f); glVertex2i( 32,  32);
  glEnd();  
  glEnable(GL_DEPTH_TEST);

  // Restore old model view matrix
  glPopMatrix();

  // Restore old projection matrix
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  UnlockMutex();
}

static void DrawScene(GLWindow *glw)
{
  // Clear the old scene
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glLoadIdentity();

  // Rotate to the correct direction
  glTranslatef(0, 0, 500); 
  glRotated(Vmy, 1., 0., 0.);
  glRotated(Vmx, 0., 1., 0.);

  // Draw the skybox
  glCallList(CLSkybox);

  // Move the player
  glTranslatef(Pos.x, Pos.y-50, Pos.z); 

  // Draw the static world
  glCallList(CLWorld);

  // Draw all the other participants
  DrawParticipants(glw);

  // Draw the water plane
  DrawWater();

  // Draw the HUD
  DrawHUD(glw);

  // Swap to display everything
  glXSwapBuffers(glw->dpy, glw->win);
}

/***************************************************************************************
 * Gui and event handler thread with helpers (bounding)
 ***************************************************************************************/

void BoundPos()
{
  // Bound against walls
  if( Pos.x >  415 ) Pos.x =  415;
  if( Pos.x < -415 ) Pos.x = -415;
  if( Pos.z >  415 ) Pos.z =  415;
  if( Pos.z < -415 ) Pos.z = -415;

  // Bound on water
  Pos.y = -WaterLevel;
}

void* GUIThread(void *arg)
{
  XEvent event;  GLWindow glw;  KeySym key;  char keys[255],d=0;  char title[256];
  int done=0,x=0,y=0;

  // Create the window and initialize it
  glw.id = 0;
  sprintf(title, "MooseChat v1.5");
  CreateGLWindow(&glw, title, Width, Height);
  InitGLWindow(&glw);

  // Build any needed call lists
  BuildSkyBox(&glw);
  BuildWorld(&glw);
  BuildAvatar(&glw);
  BuildWater(&glw);

  // Start the player up off the ground a bit and looking forward
  Pos.y = -(WaterLevel+15);
  Vmy   = -10;

  // Process events until we are "done"
  while (!done){
    // Is there an available event?
    if (XCheckIfEvent(glw.dpy, &event, AnyEvent, NULL)){
      switch (event.type) {
      case Expose:
        // Redraw
        if (event.xexpose.count != 0)
          break;
        DrawScene(&glw);
        break;
      case ConfigureNotify:
        // Resize
        if ((event.xconfigure.width != glw.width) || (event.xconfigure.height != glw.height)) {
          Width  = glw.width = event.xconfigure.width;
          Height = glw.height = event.xconfigure.height;
          ResizeGLScene(event.xconfigure.width, event.xconfigure.height);
        }
        break;
      case ClientMessage:    
        // Close button
        if (*XGetAtomName(glw.dpy, event.xclient.message_type) == *"WM_PROTOCOLS")
          done = 1;
        break;
      case KeyPress:
        // Key press
        if( XLookupString(&event.xkey, keys, 255, &key, 0) == 1 ) {
          switch (*keys){
	  case 'W':
	  case 'w':
	    LockMutex();
	    Pos.x += (float)(sin((360-Vmx)*PI/180.0f)*cos((360-Vmy)*PI/180.0f)*KeySpeed);
	    Pos.z += (float)(cos((360-Vmx)*PI/180.0f)*cos((360-Vmy)*PI/180.0f)*KeySpeed);
	    Pos.y -= (float)(sin((360-Vmy)*PI/180.0f)*KeySpeed);
	    UnlockMutex();
	    break;
	  case 'S':
	  case 's':
	    LockMutex();
	    Pos.x -= (float)(sin((360-Vmx)*PI/180.0f)*cos((360-Vmy)*PI/180.0f)*KeySpeed);
	    Pos.z -= (float)(cos((360-Vmx)*PI/180.0f)*cos((360-Vmy)*PI/180.0f)*KeySpeed);
	    Pos.y += (float)(sin((360-Vmy)*PI/180.0f)*KeySpeed);
	    UnlockMutex();
	    break;
	  case 'D':
	  case 'd':
	    LockMutex();
	    Pos.x += (float)(sin((360-90-Vmx)*PI/180.0f)*KeySpeed);
	    Pos.z += (float)(cos((360-90-Vmx)*PI/180.0f)*KeySpeed);
	    UnlockMutex();
	    break;
	  case 'A':
	  case 'a':
	    LockMutex();
	    Pos.x += (float)(sin((360+90-Vmx)*PI/180.0f)*KeySpeed);  
	    Pos.z += (float)(cos((360+90-Vmx)*PI/180.0f)*KeySpeed);
	    UnlockMutex();
            break;
          }
        }
        break;
      case ButtonRelease:
        // Mouse up
        d = 0;
        break;
      case MotionNotify:
        // Mouse move
        Mx = event.xbutton.x;
        My = event.xbutton.y;
        if(d) {
	  LockMutex();
          Vmx += event.xbutton.x-x;
          Vmy += event.xbutton.y-y;
	  UnlockMutex();
          x = event.xbutton.x;
          y = event.xbutton.y;
        }
        break;
      case ButtonPress:
        // Mouse down
        switch (event.xbutton.button){
        case 1:
          x = event.xbutton.x; 
          y = event.xbutton.y;
          d = 1;
          break;
        case 3:
	  Avatar = (Avatar+1)%nAvatars;
	  break;
        }
      }
    } else {
      // No event needs to be processed.  I guess we push frames..
      BoundPos();
      DrawScene(&glw);

      // This acts as a frame limiter
      Yeild(10000);
    }
  }
  
  /* We are done, kill the window and exit */
  KillGLWindow(&glw);
  exit(1);
  return NULL;
}

/***************************************************************************************
 * A simple sum checksum to add to UDP's built in checksum
 ***************************************************************************************/

unsigned short ComputeCheckSum(void *p, int l)
{
  unsigned short cs = 0;  int i;

  // Simple sum
  for(i=0; i<l; i++)
    cs += ((unsigned char*)p)[i];

  return htons(cs);
}

/***************************************************************************************
 * Client threads
 ***************************************************************************************/

void* ClientSendThread(void *arg)
{
  unsigned char *pkt;  unsigned short *chksum;  ClientUpdate *u;

  // Create a word aligned buffer
  if( !(pkt=(unsigned char*)malloc(sizeof(ClientUpdate)+sizeof(unsigned short))) ) {
    perror("ClientSendThread(): malloc()");
    exit(1);
  }
  u      = (ClientUpdate*)pkt;
  chksum = pkt+sizeof(ClientUpdate);

  // We do this forever
  while(1) {
    // Build a packet for the server
    LockMutex();
    u->pos.x = ntohl(Pos.x);
    u->pos.y = ntohl(Pos.y);
    u->pos.z = ntohl(Pos.z);
    memcpy(u->text, Text, 32);
    memcpy(u->name, Name, 8);
    u->avatar = Avatar;
    *chksum = ComputeCheckSum(u, sizeof(ClientUpdate));
    UnlockMutex();

    // Send off the packet
    if( Sendto(Sock, u, sizeof(ClientUpdate)+sizeof(unsigned short), 0, (struct sockaddr *)&SrvAddr, AddrLen) < 0 ) {
      perror("ClientSendThread(): Sendto()");
      exit(1);
    } 
    ClientSockBound = 1;
    nPacketsOut++;
    nBytesOut += sizeof(ClientUpdate)+sizeof(unsigned short);

    // Periodic updates are sent every:
    Yeild(100000);
  }
  
  // Just to shut up the compiler
  return NULL;
}

void* ClientRecvThread(void *arg)
{
  struct sockaddr_in from_addr;  int addr_len,pl,i;  unsigned char *pkt;  ClientUpdate *u;  GlobalUpdateHdr *g;
  unsigned short chksum;

  // Create a word aligned buffer
  if( !(pkt=(unsigned char*)malloc(1024)) ) {
    perror("ClientRecvThread(): malloc()");
    exit(1);
  }
  g = (GlobalUpdateHdr*)pkt;

  // Don't start util we have at least sent one packet.
  // Polling here is a little lame, but it's a simple solution.
  while( !ClientSockBound ) 
    Yeild(100000);

  // We do this forever
  while(1) {
    // Read a packet from the server
    addr_len = sizeof(struct sockaddr_in);
    if( (pl = Recvfrom(Sock, pkt, 1024, 0, (struct sockaddr *)&from_addr, &addr_len)) < 0 ) {
      perror("ClientRecvThread(): Recvfrom()");
      exit(1);
    }
 
    // Check packet length
    if( pl < (sizeof(GlobalUpdateHdr)+sizeof(ClientUpdate)) ) {
      fprintf(stderr, "ClientRecvThread(): recv_size < sizeof(GlobalUpdateHdr)+sizeof(ClientUpdate)\n");
      continue;
    }
    // Check the checksum
    chksum    = g->chksum;
    g->chksum = 0;
    if( chksum != ComputeCheckSum(g, pl) ) {
      fprintf(stderr, "ClientRecvThread(): checksum missmatch\n");
      continue;
    }
    // Make sure g->n <= MAX_PARTICIPANTS
    if( g->n > MAX_PARTICIPANTS ) {
      fprintf(stderr, "ClientRecvThread(): g->n > MAX_PARTICIPANTS\n");
      continue;
    }
    nPacketsIn++;
    nBytesIn += pl;

    // Replace our entire participant array with the new update data
    LockMutex();
    WaterLevel    = g->wl;
    nParticipants = g->n;
    for(u=(ClientUpdate*)(pkt+sizeof(GlobalUpdateHdr)),i=0; i<nParticipants; i++,u++) {
      // Trim any strings to be on the safe side
      u->text[31] = 0;
      u->name[7]  = 0;
      // At this point, we have an index of a participant that needs to be updated
      memcpy(&Participants[i].name, u->name, 8);
      memcpy(&Participants[i].text, u->text, 32);
      Participants[i].pos.x  = ntohl(u->pos.x);
      Participants[i].pos.y  = ntohl(u->pos.y);
      Participants[i].pos.z  = ntohl(u->pos.z);
      Participants[i].avatar = u->avatar;
    }
    UnlockMutex();
  }
}

/***************************************************************************************
 * Server threads and helpers
 ***************************************************************************************/

void BoundParticipants()
{
  int i;

  for(i=0; i<nParticipants; i++)
    Participants[i].pos.y = -70+5*sin(Participants[i].ticks/5.);
}

void* ServerSendThread(void *arg)
{
  unsigned char *pkt;  int i,ticks;  ClientUpdate *u;  GlobalUpdateHdr *g;

  // Create a word aligned buffer
  if( !(pkt=(unsigned char*)malloc(1024)) ) {
    perror("ServerSendThread(): malloc()");
    exit(1);
  }
  g = (GlobalUpdateHdr*)pkt;

  // We do this forever
  while(1) {
    LockMutex();

    // Update any animated objects that need it
    WaterLevel = 70+5*sin(ticks/5.);
    BoundParticipants();

    // Build an update packet for all clients
    g->n = nParticipants+1; 
    // Three bytes of padding follow nParticipants
    for(u=(ClientUpdate*)(pkt+sizeof(GlobalUpdateHdr)),i=0; i<nParticipants; i++,u++) {
      u->pos.x = htonl(Participants[i].pos.x);
      u->pos.y = htonl(Participants[i].pos.y);
      u->pos.z = htonl(Participants[i].pos.z);
      memcpy(u->text, &Participants[i].text, 32);
      memcpy(u->name, &Participants[i].name, 8);
      u->avatar = Participants[i].avatar;
    }
    // Add self to the list
    u->pos.x = htonl(Pos.x);
    u->pos.y = htonl(Pos.y);
    u->pos.z = htonl(Pos.z);
    memcpy(u->text, &Text, 32);
    memcpy(u->name, &Name, 8);
    u->avatar = Avatar;
    UnlockMutex();

    // Send the packet off to all the clients
    for(i=0; i<nParticipants; i++) {
      g->wl     = 70+5*sin(Participants[i].ticks/5.);
      g->chksum = 0;
      g->chksum = ComputeCheckSum(g, sizeof(GlobalUpdateHdr)+((nParticipants+1)*sizeof(ClientUpdate)));
      if( Sendto(Sock, pkt, sizeof(GlobalUpdateHdr)+((nParticipants+1)*sizeof(ClientUpdate)), 0, 
		 (struct sockaddr *)&Participants[i].saddr, Participants[i].saddr_len) < 0 ) {
	perror("ServerSendThread(): Sendto()");
	exit(1);
      } 
      Participants[i].ticks++;
      nPacketsOut++;
      nBytesOut += sizeof(GlobalUpdateHdr)+((nParticipants+1)*sizeof(ClientUpdate));
    }

    // Periodic updates are sent every:
    Yeild(100000);
    ticks++;
  }
  
  // Just to shut up the compiler
  return NULL;
}

void* ServerRecvThread(void *arg)
{
  struct sockaddr_in from_addr;  int addr_len,pl,i;  ClientUpdate *u;  unsigned short *chksum;

  // Create a word aligned buffer
  if( !(u=(ClientUpdate*)malloc(sizeof(ClientUpdate)+sizeof(unsigned short))) ) {
    perror("ServerRecvThread(): malloc()");
    exit(1);
  }
  chksum = ((unsigned char*)u)+sizeof(ClientUpdate);
    
  // We do this forever
  while(1) {
    // Read a packet from any client
    addr_len = sizeof(struct sockaddr_in);
    if( (pl = Recvfrom(Sock, u, sizeof(ClientUpdate)+sizeof(unsigned short), 
		       0, (struct sockaddr *)&from_addr, &addr_len)) < 0 ) {
      perror("ServerRecvThread(): Recvfrom()");
      exit(1);
    }

    // Check the packet size
    if( pl != sizeof(ClientUpdate)+sizeof(unsigned short) ) {
      fprintf(stderr, "ServerRecvThread(): recv_size != sizeof(ClientUpdate)+sizeof(unsigned short)\n");
      continue;
    }
    // Check the packet checksum
    if( *chksum != ComputeCheckSum(u, sizeof(ClientUpdate)) ) {
      fprintf(stderr, "ServerRecvThread(): checksum missmatch\n");
      continue;
    }
    nPacketsIn++;
    nBytesIn += pl;
   
    // Trim any strings to be on the safe side
    u->text[31] = 0;
    u->name[7]  = 0;
    // See if we can find a participant to update
    LockMutex();
    for(i=0; i<nParticipants; i++)
      if( !strcmp(Participants[i].name,u->name) )
	break;
    // Need to create a new particitpant?
    if( i == nParticipants ) {
      if( nParticipants == MAX_PARTICIPANTS ) {
	// Full, drop update packet
	continue;
      }
      i = nParticipants++;
      printf("NewUser: \"%s\"\n", u->name);
    }
    // At this point, we have an index of a participant that needs to be updated
    memcpy(&Participants[i].saddr, &from_addr, addr_len);
    Participants[i].saddr_len = addr_len;
    memcpy(&Participants[i].name, u->name, 8);
    memcpy(&Participants[i].text, u->text, 32);
    Participants[i].pos.x  = ntohl(u->pos.x);
    Participants[i].pos.y  = ntohl(u->pos.y);
    Participants[i].pos.z  = ntohl(u->pos.z);
    Participants[i].avatar = u->avatar;
    BoundParticipants();
    UnlockMutex();
  }
}

/***************************************************************************************
 * Application entry point and helpers
 ***************************************************************************************/

void StartThreads()
{
  pthread_attr_t a;  pthread_t t;

  // Create a mutex for thread sync
  pthread_mutex_init((pthread_mutex_t*)&Mutex, NULL);

  // Set up the thread attributes
  pthread_attr_init(&a);
  pthread_attr_setdetachstate(&a, PTHREAD_CREATE_DETACHED);
  pthread_attr_setscope(&a, PTHREAD_SCOPE_SYSTEM);

  // Create the GUI thread
  if( pthread_create(&t, &a, GUIThread, NULL) ) {
    perror("StartThreads(): pthread_create(GUIThread)");
    exit(1);
  }

  // Start any needed netowrk threads
  if( Mode == CLIENT ) {
    // Client only
    if( pthread_create(&t, &a, ClientRecvThread, NULL) ) {
      perror("StartThreads(): pthread_create(ClientRecvThread)");
      exit(1);
    }
    if( pthread_create(&t, &a, ClientSendThread, NULL) ) {
      perror("StartThreads(): pthread_create(ClientSendThread)");
      exit(1);
    }
  } else {
    // Server only
    if( pthread_create(&t, &a, ServerRecvThread, NULL) ) {
      perror("StartThreads(): pthread_create(ServerRecvThread)");
      exit(1);
    }
    if( pthread_create(&t, &a, ServerSendThread, NULL) ) {
      perror("StartThreads(): pthread_create(ServerSendThread)");
      exit(1);
    }
  }

}

void UseageError() 
{
  printf("Useage:\n\tmchat <S|C> <IP address> <port> <alias>\n");
  exit(1);
}

int main(int argc, char **argv)
{
  int port;  struct in_addr ip;

  // Check command line args...
  if( argc < 5 )
    UseageError();
  // Mode
  if( !strcmp(argv[1],"C") ) {
    Mode = CLIENT;
  } else if( !strcmp(argv[1],"S") ) {
    Mode = SERVER;
  } else {
    UseageError();
  }
  // IP
  if( inet_pton(AF_INET, argv[2], &ip) < 1 ) {
    UseageError();
  }
  // Port
  if( sscanf(argv[3], "%d", &port) != 1 ) {
    UseageError();
  }
  // Copy the alias
  memcpy(Name, argv[4], (((strlen(argv[4])+1)>8)?(8):(strlen(argv[4])+1)));
  Name[7] = 0;

  // Initialize the network socket and state
  AddrLen = sizeof(struct sockaddr_in);
  SrvAddr.sin_addr.s_addr = ip.s_addr;
  SrvAddr.sin_family      = PF_INET;
  SrvAddr.sin_port        = htons(port);
  if((Sock = Socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
    perror("main(): Socket()");
    exit(1);
  }
  // Bind socket
  if( Mode == SERVER ) {
    if((Bind(Sock, (struct sockaddr *)&SrvAddr, AddrLen)) < 0)
      perror("main(): Bind()");
  }
  // I honestly don't know if you can bind() to an ephemeral IP and port.
  // That's what I really want to do here.  I have a hack around this 
  // in the client's network thread section.
  
  // Start threads
  StartThreads();
  
  // Kill off the current thread
  pthread_exit(0);
  return 0;
}
