#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "casim1d.h"
#define STATE_FROM_EPS_C
#include "statefromeps.h"


static void* Malloc(size_t size)
{
  void *ptr;

  if( !(ptr=malloc(size)) ) {
    fprintf(stderr,"Failed to allocate space while reading in an EPS file.\n");
    exit(1);
  }

  return ptr;
}


state_t* statefromeps(char *fn)
{
  FILE    *f;
  state_t *state;
  char     line[8192],*rule,*t;
  int      i,r,k,x,y,v,w,h,nr,My=0;
  struct timeval tv;

  // Open EPS file
  if( !(f=fopen(fn,"r")) ) {
    return NULL;
  }

  // Allocate a new state object
  state = Malloc(sizeof(state_t));
  memset(state, 0, sizeof(state_t));
 
  // Process one line at a time
  while( fgets(line,sizeof(line),f) ) {
    if( (t=strstr(line,"1dcasim_hdr:")) ) {
      // Header line
      if( sscanf(t+strlen("1dcasim_hdr:"),"r-%d-K-%d-w-%d-h-%d-rules",&r,&k,&w,&h) != 4 ) {
	fprintf(stderr,"Failed to parse EPS header line.\n");
	return NULL;
      }
      state->r = r;
      state->K = k;
      state->cellsx = w;
      state->cellsy = h;
      state->cells = Malloc(h*sizeof(int*));
      for(i=0; i<h; i++) {
	state->cells[i] = Malloc(w*sizeof(int));
      }
      // Calculate size from header prefix
      nr = (2*r+1)*(k-1)+1;
      state->rules  = Malloc(nr*sizeof(int));
      state->nrules = nr;
      // Read rules from header
      if( (t=strstr(line,"--")) ) {
	rule = strtok(t+2, "-");
	for(i=0; i<nr && rule; ++i) {
	  state->rules[i] = atoi(rule);
	  rule = strtok(NULL, "-");
	}
      } else {
	fprintf(stderr,"Failed to parse EPS rule header line.\n");
	return NULL;
      }
    } else if( (t=strstr(line,"1dcasim_cell:")) ) {
      // Cell line
      if( !state->rules ) {
	fprintf(stderr,"Failed to find EPS header line before cell line.\n");
	return NULL;
      }
      // Fill in the cell
      if( sscanf(t+strlen("1dcasim_cell:"),"%d,%d:%d",&x,&y,&v) != 3 ) {
	fprintf(stderr,"Failed to read EPS cell line.\n");
	return NULL;
      }
      state->cells[y][x] = v;
      // Record the highest cell
      if( y > My ) {
      	My = y;
      }
    }
  }

  // Assume image is full from 0,0 to w,My
  state->time = My;

  // Initialize the random number generator
  if( gettimeofday(&tv, NULL) == -1 ) {
    random_initrand(&state->random,7);
  } else {
    random_initrand(&state->random,tv.tv_usec);
  }
  
  // Return the newly created state
  return state;
}
