#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <math.h>

#include "types.h"
#include "util.h"
#include "random.h"
#include "params.h"
#include "io_epsnn.h"
#include "io_bitmap.h"
#define ONEDCASIM_C
#include "casimnn.h"
#include "statsnn.h"
#include "gui.h"


static state_t  *State = NULL;
pthread_mutex_t  StateLock = PTHREAD_MUTEX_INITIALIZER;

// Precompute neighborinfo for speed
typedef struct str_p2_t {
  int x;
  int y;
} p2_t;

typedef struct str_neighbor_t {
  p2_t *r1nbrs;
  int   nr1nbrs;
  p2_t *r2nbrs;
  int   nr2nbrs;
} neighbor_t;


// !!av: a hack only for images in batch mode
char *epsfn = NULL;


////////////////////////////////////////////////////////////
// Sets up the global state to get ready for play
////////////////////////////////////////////////////////////


void static InitW(int **patterns, int npatterns)
{  
  float sum;
  int   i,j,k;

  // Allocate and clear
  if( !State->W ) {
    State->W = malloc(State->w*State->h*sizeof(float*));
    for(i=0; i<State->w*State->h; i++) {
      State->W[i] = malloc(State->w*State->h*sizeof(float));
      memset(State->W[i],0,State->w*State->h*sizeof(float));
    }
  }

  // Fill in W matrix
  for(i=0; i<State->h*State->w; i++) {
    for(j=0; j<State->h*State->w; j++) {
      // Ignore the diagonal
      if( i != j ) {
	sum = 0;
	for(k=0; k<npatterns; k++) {
	  sum += patterns[k][i]*patterns[k][j];
	}
	State->W[i][j] = sum / (State->w*State->h);
      } else {
	// Self-coupling
	State->W[i][j] = 0.0f;
      }
    }
  }
}


static void init_state(int w, int h, int bch)
{
  struct timeval tv;
  int            i,j,r;
  
  // Allocate the main state structure
  if( !(State=malloc(sizeof(state_t))) ) {
    Error("Could not allocate state structure (%u)\n",sizeof(state_t));
  }

  // Go ahead and clear the memory to start with.
  memset(State,0,sizeof(state_t));

  // Do we need to run in batch mode as fast as possible with no GUI?
  State->bch = bch;

  // Initialize the random number generator
  if( gettimeofday(&tv, NULL) == -1 ) {
    Warn("gettiemofday() failed; seeding random generator with 7 instead.\n");
    random_initrand(&State->random,7);
  } else {
    random_initrand(&State->random,tv.tv_usec);
  }

  // Set the current time
  State->time = 0;
  if( bch ) {
    State->rolling = 1;
  }

  // Allocate cells
  State->w = w;
  State->h = h;
  State->cells = malloc(State->h*sizeof(int*));
  for(i=0; i<State->h; i++) {
    State->cells[i] = malloc(State->w*sizeof(int));
    memset(State->cells[i], 0, State->w*sizeof(int));
  }  

  // Give a random starting state
  for(j=0; j<State->h; j++) {
    for(i=0; i<State->w; i++) {
      r = random_U01(&State->random);
      if( r < 0.5 ) {
	State->cells[j][i] = -1;
      } else {
	State->cells[j][i] = +1;
      }
    }
  }

}

void write_cell(int x, int y, int v)
{
  int i,j;

  pthread_mutex_lock(&StateLock);  
  for(i=-1; i<=1; i++) {
    for(j=-1; j<=1; j++) {
      if( ((y+i) > 0) && ((x+j) > 0) && ((y+i) < State->h) && ((x+j) < State->w) ) {
	State->cells[y+i][x+j] = v;
      }
    }
  }

  pthread_mutex_unlock(&StateLock);
}

void set_state()
{
  int i,j;
 
  pthread_mutex_lock(&StateLock);  

  // Give a random starting state
  for(j=0; j<State->h; j++) {
    for(i=0; i<State->w; i++) {
      if( random_U01(&State->random) < 0.5 ) {
	State->cells[j][i] = -1;
      } else {
	State->cells[j][i] = +1;
      }
    }
  }

  // Set the current time
  State->time = 0;
  State->converged = 0;

  pthread_mutex_unlock(&StateLock);
}

void set_pattern(int *pattern)
{
  int i;

  pthread_mutex_lock(&StateLock);  

  for(i=0; i<State->h; i++) {
    memcpy(State->cells[i],pattern+i*State->w,State->w*sizeof(int));
  }

  pthread_mutex_unlock(&StateLock);
}

void decimate()
{
  int i,j,k;

  pthread_mutex_lock(&StateLock);  

  for(k=0; k<State->w*State->h*0.2; k++) {
    i = random_rnd(&State->random,State->h);
    j = random_rnd(&State->random,State->w);
    
    // Decimate by flipping one random neron / cell
    if( random_U01(&State->random) < 0.5 ) {
      State->cells[i][j] = -1;
    } else {
      State->cells[i][j] = +1;
    }
  }
 
 pthread_mutex_unlock(&StateLock);
}


int run_state()
{
  int rv;

  pthread_mutex_lock(&StateLock);  
  State->rolling = (State->rolling == 0);
  rv = State->rolling;
  pthread_mutex_unlock(&StateLock);

  return rv;
}



////////////////////////////////////////////////////////////
// Main game loop and helpers
////////////////////////////////////////////////////////////

void handle_convergence()
{
#if 0
  // If we are in batch mode; output the data
  if( State->bch ) {
    printf("#(%d,%d,%d,%d,%0.4f,%0.4f,%0.4f) %d %0.4f %0.4f %0.4f\n",
	   State->w,
	   State->h,
	   State->r1,
	   State->r2,
	   State->j1,
	   State->j2,
	   State->hv,
	   State->characteristic_correlation_len,
	   State->spatial_correlation[State->characteristic_correlation_len],
	   State->joint_entropy[State->characteristic_correlation_len],
	   State->mutual_information[State->characteristic_correlation_len]);
    for(i=0; i<maxlen; i++) {
      printf("(%d,%d,%d,%d,%0.4f,%0.4f,%0.4f) %d %0.4f %0.4f %0.4f\n",
	     State->w,
	     State->h,
	     State->r1,
	     State->r2,
	     State->j1,
	     State->j2,
	     State->hv,
	     i,
	     State->spatial_correlation[i],
	     State->joint_entropy[i],
	     State->mutual_information[i]);
    }
    fflush(stdout);
    // Write eps image if needed
    if( epsfn ) {
      io_epsfromstate(State, epsfn, 1.0f, 1.0f);
    }
    exit(0);
  } 
#else
  // !! stub
  State->converged = 1;
  printf("Converged.\n");
#endif
}
  

int progress_ca()
{
  float sum;
  int   i,j,k,l,ov;

  // We process a single random (i,j) cell at a time
  i = random_rnd(&State->random,State->h);
  j = random_rnd(&State->random,State->w);
  ov = State->cells[i][j];

  // Compute local field (compute against all other neurons / cells)
  sum = 0;
  for(k=0; k<State->h; k++) {
    for(l=0; l<State->w; l++) {
      sum += State->W[i*State->w+j][k*State->w+l] * State->cells[k][l];
    }
  }

  // Handle local field sum
  if( sum < 0 ) {
    State->cells[i][j] = -1;
  } else {
    State->cells[i][j] = +1;
  }

  // Return the old value
  return (ov==State->cells[i][j]);
}


void tick()
{ 
  static int match_count=0;

  if( State->time == 0 ) {
    match_count = 0;
  }

  if( State->rolling && !(State->converged >= 2) ) {
    // Process one step of the CA
    if( progress_ca() ) {
      // No cell change
      match_count++;
    } else {
      // The cell changed
      match_count = 0;
    }
    State->time++;
  }

  // Estiname convergence
  if( match_count > 10000 ) {
    // Convergence has been reached
    if( !State->converged ) {
      handle_convergence();
    }
  }
}


u64b_t get_time()
{
  struct timeval tv;

  gettimeofday(&tv, NULL);

  return tv.tv_sec * 1000000ull + tv.tv_usec;
}


static void simulator_loop()
{
  u64b_t i,t1,t2,sleep;

  if( !State->bch ) {
    // Start the GUI
    StartGUI(VERSION_STR,((gstate_t*)State)); 
    UpdateGuiState(((gstate_t*)State));
  }

  while( 1 ) {
    pthread_mutex_lock(&StateLock);

    // Progress the game one time step
    t1 = get_time();

    for(i=0; i<(State->w+State->h)/2; i++) {
      tick();
    }
    
    pthread_mutex_unlock(&StateLock);

    if( !State->bch ) {
      // Update the gui
      UpdateGuiState(((gstate_t*)State));
    }

    t2 = get_time();

    // Sleep for the remainder of the tick cycle
    if( !State->bch ) {
      // Only in non-batch mode do we slow down for GUI, etc.
      if( (t1 < t2) && ((t2-t1) < 2000) ) {
	sleep = 2000 - (t2-t1);
	if( sleep > 0 ) {
	  usleep( sleep );
	}
      }
    }

  }

}


void print_usage_and_exit(int status)
{
  fprintf(stderr,
	  "Usage: casim1d [OPTION]...\n"
	  "Hopfield Neural Network Simulator (c)2012 Aaron Vose and Paul Giblock\n"
	  "\n"
	  "Available Options:\n"
	  "\n"
	  "  -w  CELLS              Width of cells               (Default: %d)\n"
	  "  -h  CELLS              Height of cells              (Default: %d)\n"
	  "\n"
	  "  -f  FILENAME           EPS file from which to initialize\n"
	  "  -p  FILE, ...          BMP files of patterns to embed in NN\n"
	  "\n"
	  "  -?, --help            Print the help message and exit\n",
	  PARAM_w, PARAM_h);

  exit(status);
}
 
char** parse_array_option(char *str, char *name, int *np)
{
  char **pn=NULL,*l;
  int    n=0;
  
  if (str) {
    while( (l=strtok(((n)?(NULL):(str)),",\0")) ) {
      pn = realloc(pn,(n+1)*sizeof(char*));
      pn[n] = strdup(l);
      n++;
    }
    if( n == 0 ) {
      fprintf(stderr, "-%s requires a valid comma separated list of file names.\n\n", name);
      print_usage_and_exit(1);
    }
  }

  *np = n;
  return pn;
}

int parse_int_option(char *str, char *name)
{
  int val;
  
  if (str) {
    if (sscanf(str, "%d", &val) != 1) {
      fprintf(stderr, "-%s requires a valid integer argument.\n\n", name);
      print_usage_and_exit(1);
    }
  } else {
    fprintf(stderr, "-%s option requires an argument.\n\n", name);
    print_usage_and_exit(1);
  }
  return val;
}
 
 
float parse_float_option(char *str, char *name)
{
  float val;

  if (str) {
    if (sscanf(str, "%f", &val) != 1) {
      fprintf(stderr, "-%s requires a valid integer argument.\n\n", name);
      print_usage_and_exit(1);
    }
  } else {
    fprintf(stderr, "-%s option requires an argument.\n\n", name);
    print_usage_and_exit(1);
  }

  return val;
}


////////////////////////////////////////////////////////////
// Application entry point
////////////////////////////////////////////////////////////


static void fill_pattern(char *fn, int *pattern)
{
  io_bitmap_t bmp;
  int         i,j;

  
  // Read bitmap
  io_bitmap_load(fn, &bmp);

  // Make sure sizes match
  if( (bmp.w != State->w) || (bmp.h != State->h) ) {
    fprintf(stderr,
	    "Bitmap size in pixels does not match neural network size (%d,%d) != (%d,%d).\n",
	    bmp.w,bmp.h,State->w,State->h);
    exit(1);
  }
  
  // If the pixel is black, I will say p = -1, else i'll say p = 1;
  for(i=0; i<bmp.h; i++) {
    for(j=0; j<bmp.w; j++) {
      if( (bmp.d[(i*bmp.w+j)*3+0]+bmp.d[(i*bmp.w+j)*3+1]+bmp.d[(i*bmp.w+j)*3+2]) == 0 ) {
	pattern[i*bmp.w+j] = -1;
      } else {
	pattern[i*bmp.w+j] = +1;
      }
    }
  }

  // Cleanup to be nice
  io_bitmap_free(&bmp);
}


int main(int argc, char *argv[])
{
  float  r;
  char  *opt, *val;
  int    i,j,a;

  // Default option
  int    w   = PARAM_w;
  int    h   = PARAM_h;
  int    bch = PARAM_batch;
  char **p   = NULL;
  int    np  = 0;

  // Parse inputs
  for(a=1; a<argc; ++a) {
    // Help
    if (strcmp(argv[a], "--help") == 0 || strcmp(argv[a], "-?") == 0) {
      print_usage_and_exit(0);
    }
    // a -X param( 
    else if( ((strlen(argv[a])==2 || strlen(argv[a])==3) || strlen(argv[a])==4) && argv[a][0]=='-') {
      opt = &(argv[a][1]);
      // Only use val if not last argument
      val = (a==argc-1) ? NULL : argv[a+1];
      if( !strcmp("eps", opt) ||
	  !strcmp("PS", opt) ||
	  !strcmp("ps", opt) ||
	  !strcmp("EPS", opt) ) {
	// !!av: EPS IO not working yet
	//if( val ) {
	//  epsfn = strdup(val);
	//}
	//a++;
      } else if( !strcmp("B", opt) ||
		 !strcmp("b", opt) ) {
	bch = 1;
      } else if( !strcmp("h", opt) ||
		 !strcmp("H", opt) ) {
	h = parse_int_option(val, "h");
	a++;
      } else if( !strcmp("P", opt) ||
		 !strcmp("p", opt) ) {
	p = parse_array_option(val, "p", &np);
	a++;
      } else if( !strcmp("w", opt) ||
		 !strcmp("W", opt) ) {
	w = parse_int_option(val, "w");
	a++;
      } else if( !strcmp("f", opt) ||
		 !strcmp("F", opt) ) {
	// !!av: EPS IO not working yet
	// Input EPS file
	// State = io_epstostate(val);
	// a++;
      }  else {
	fprintf(stderr, "Unknown option '-%s'.\n\n", opt);
	print_usage_and_exit(1);
      }
    }
  }

  // Get ready to do stuffs...
  if( !State ) {
    // The state may have already been set from a file.
    init_state(w, h, bch);
  }

  // Get patterns ready
  if( np ) {
    // Store patterns from files given on command line
    State->npatterns = np;
    State->patterns = malloc(State->npatterns*sizeof(int*));
    for(i=0; i<State->npatterns; i++) {
      State->patterns[i] = malloc(State->w*State->h*sizeof(int));
      fill_pattern(p[i], State->patterns[i]);
    }
  } else {
    // No inputs so store random patterns?!?
    State->npatterns = np = 10;
    State->patterns = malloc(State->npatterns*sizeof(int*));
    for(i=0; i<State->npatterns; i++) {
      State->patterns[i] = malloc(State->w*State->h*sizeof(int));
      for(j=0; j<State->w*State->h; j++) {
	r = random_U01(&State->random);
	if( r < 0.5 ) {
	  State->patterns[i][j] = -1;
	} else {
	  State->patterns[i][j] = +1;
	}
      }
    }
  }
  set_pattern(State->patterns[0]);

  // Init weight matrix
  InitW(State->patterns, State->npatterns);

  // Game loop
  simulator_loop();

  // Done, exit game.
  return 0;
}

