#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <math.h>

#include "types.h"
#include "util.h"
#include "random.h"
#include "params.h"
#include "io_eps2d.h"
#define ONEDCASIM_C
#include "casim2d.h"
#include "stats2d.h"
#include "gui.h"


static state_t  *State = NULL;
pthread_mutex_t  StateLock = PTHREAD_MUTEX_INITIALIZER;

// Precompute neighborinfo for speed
typedef struct str_p2_t {
  int x;
  int y;
} p2_t;

typedef struct str_neighbor_t {
  p2_t *r1nbrs;
  int   nr1nbrs;
  p2_t *r2nbrs;
  int   nr2nbrs;
} neighbor_t;

static neighbor_t **Neighbors = NULL;

// !!av: a hack only for images in batch mode
char *epsfn = NULL;


////////////////////////////////////////////////////////////
// Sets up the global state to get ready for play
////////////////////////////////////////////////////////////

void indexmap(int x, int y, int *nx, int *ny)
{
  int i;

  i = x;
  // Catch the case where i is too small
  while( i < 0 ) {
    i = State->w + i;
  }
  // Catch the case where i is too large
  while( i >= State->w ) {
    i = i - State->w;
  }
  // Record
  *nx = i;


  i = y;
  // Catch the case where i is too small
  while( i < 0 ) {
    i = State->h + i;
  }
  // Catch the case where i is too large
  while( i >= State->h ) {
    i = i - State->h;
  }
  // Record
  *ny = i;
}

static void Compute_Neighbors()
{
  float d;
  int   i,x,y,xo,yo,tx,ty;

  // Allocate neighbor cache
  Neighbors = malloc(State->h*sizeof(neighbor_t*));
  for(i=0; i<State->h; i++) {
    Neighbors[i] = malloc(State->w*sizeof(neighbor_t));
    memset(Neighbors[i],0,State->w*sizeof(neighbor_t));
  }

  // Precache for all cells
  for(y=0; y<State->h; y++) {
    for(x=0; x<State->w; x++) {
      // Consider all offsets from each cell
      for(xo=-State->r2; xo<=State->r2; xo++) {
	for(yo=-State->r2; yo<=State->r2; yo++) {
	  d = fabs(xo) + fabs(yo);
	  if( d < State->r1 ) {
	    // Neighbor is in promotional radius
	    Neighbors[y][x].nr1nbrs++;
	    Neighbors[y][x].r1nbrs = realloc(Neighbors[y][x].r1nbrs, Neighbors[y][x].nr1nbrs*sizeof(p2_t));
	    indexmap(x+xo, y+yo, &tx, &ty);
	    Neighbors[y][x].r1nbrs[Neighbors[y][x].nr1nbrs-1].x = tx;
	    Neighbors[y][x].r1nbrs[Neighbors[y][x].nr1nbrs-1].y = ty;
	  } else if( d < State->r2 ) {
	    // Neighbor is in inhibitory radius
	    Neighbors[y][x].nr2nbrs++;
	    Neighbors[y][x].r2nbrs = realloc(Neighbors[y][x].r2nbrs, Neighbors[y][x].nr2nbrs*sizeof(p2_t));
	    indexmap(x+xo, y+yo, &tx, &ty);
	    Neighbors[y][x].r2nbrs[Neighbors[y][x].nr2nbrs-1].x = tx;
	    Neighbors[y][x].r2nbrs[Neighbors[y][x].nr2nbrs-1].y = ty;
	  }
	}
      }
    }
  }

}


static void init_state(int w, int h, int r1, int r2, float j1, float j2, float hv, int bch)
{
  struct timeval tv;
  int            i,j;
  
  // Allocate the main state structure
  if( !(State=malloc(sizeof(state_t))) ) {
    Error("Could not allocate state structure (%u)\n",sizeof(state_t));
  }

  // Go ahead and clear the memory to start with.
  memset(State,0,sizeof(state_t));

  // Do we need to run in batch mode as fast as possible with no GUI?
  State->bch = bch;

  // Go ahead and start with passed in defaults
  State->hv = hv;
  State->r1 = r1;
  State->r2 = r2;
  State->j1 = j1;
  State->j2 = j2;

  // Initialize the random number generator
  if( gettimeofday(&tv, NULL) == -1 ) {
    Warn("gettiemofday() failed; seeding random generator with 7 instead.\n");
    random_initrand(&State->random,7);
  } else {
    random_initrand(&State->random,tv.tv_usec);
  }

  // Set the current time
  State->time = 0;
  if( bch ) {
    State->rolling = 1;
  }

  // Allocate cells
  State->cellsx = State->w = w;
  State->cellsy = State->h = h;
  State->cells = malloc(State->cellsy*sizeof(int*));
  for(i=0; i<State->cellsy; i++) {
    State->cells[i] = malloc(State->cellsx*sizeof(int));
    memset(State->cells[i], 0, State->cellsx*sizeof(int));
  }  

  // Give a random starting state
  for(j=0; j<State->cellsy; j++) {
    for(i=0; i<State->cellsx; i++) {
      if( random_U01(&State->random) < 0.5 ) {
	State->cells[j][i] = -1;
      } else {
	State->cells[j][i] = +1;
      }
    }
  }
}


void set_state(int n)
{
  int i,j;
 
  pthread_mutex_lock(&StateLock);  

  // Give a random starting state
  for(j=0; j<State->cellsy; j++) {
    for(i=0; i<State->cellsx; i++) {
      if( random_U01(&State->random) < 0.5 ) {
	State->cells[j][i] = -1;
      } else {
	State->cells[j][i] = +1;
      }
    }
  }

  // Go ahead and start with some reasonable random values
  if( n ){
    State->hv  = random_U(&State->random,3);
    State->hv *= random_U(&State->random,2);
    State->hv += 0;
    State->r1  = random_U(&State->random,3);
    State->r1 *= random_U(&State->random,2);
    State->r1 += 1;
    State->r2  = State->r1;
    State->r2 += random_U(&State->random,7);
    State->r2 *= random_U(&State->random,2);
    State->r2 += 1;
    State->j1 = random_U(&State->random,1) * 2.0;
    State->j2 = random_U(&State->random,1) / 5.0;
  }

  // Set the current time
  State->time = 0;
  State->converged = 0;

  pthread_mutex_unlock(&StateLock);
}


void decimate()
{
  pthread_mutex_lock(&StateLock);  

  // Decimate by decreasing the constant offset
  State->hv *= 0.01;
  State->converged = 0;

  pthread_mutex_unlock(&StateLock);
}


int run_state()
{
  int rv;

  pthread_mutex_lock(&StateLock);  
  State->rolling = (State->rolling == 0);
  rv = State->rolling;
  pthread_mutex_unlock(&StateLock);

  return rv;
}



////////////////////////////////////////////////////////////
// Main game loop and helpers
////////////////////////////////////////////////////////////

void handle_convergence()
{
  int   i,maxlen,cl;
  float d,cd;

  // Set the converged flag so we know we are starting calcs
  // Tell the GUI too
  State->converged = 1;
  if( !State->bch ) {
    pthread_mutex_unlock(&StateLock);
    UpdateGuiState(((gstate_t*)State));
    pthread_mutex_lock(&StateLock);
  }

  // Find max length for correlations, etc.
  maxlen = ((State->w>=State->h)?(State->w/2):(State->h/2));

  // Allocate space now that we know the max length
  State->spatial_correlation = malloc((maxlen)*sizeof(float));
  State->joint_entropy       = malloc((maxlen)*sizeof(float));
  State->mutual_information  = malloc((maxlen)*sizeof(float));

  // Fill in all stats for all of the lengths in bounds
  for(i=0; i<maxlen; i++) {
    fflush(stdout);
    State->spatial_correlation[i] = stats2d_p(State, i);
    fflush(stdout);
    State->joint_entropy[i]       = stats2d_H(State, i);
    fflush(stdout);
    State->mutual_information[i]  = stats2d_I(State, i);
  }

  // Find the characteristic correlation length
  cl = -1;
  cd = 1000000; // Close enough to max_int for my use
  for(i=0; i<maxlen; i++) {
    d = fabs(State->spatial_correlation[i] - State->spatial_correlation[0]/M_E);
    // Better than current correclation length?
    if( d < cd ) {
      cd = d;
      cl = i;
    }
  }
  State->characteristic_correlation_len = cl;

  // Set the converged flag so we know we are done with convergence calcs
  State->converged = 2;
  if( !State->bch ) {
    pthread_mutex_unlock(&StateLock);
    UpdateGuiState(((gstate_t*)State));
    pthread_mutex_lock(&StateLock);
  }

  // If we are in batch mode; output the data
  if( State->bch ) {
    printf("#(%d,%d,%d,%d,%0.4f,%0.4f,%0.4f) %d %0.4f %0.4f %0.4f\n",
	   State->w,
	   State->h,
	   State->r1,
	   State->r2,
	   State->j1,
	   State->j2,
	   State->hv,
	   State->characteristic_correlation_len,
	   State->spatial_correlation[State->characteristic_correlation_len],
	   State->joint_entropy[State->characteristic_correlation_len],
	   State->mutual_information[State->characteristic_correlation_len]);
    for(i=0; i<maxlen; i++) {
      printf("(%d,%d,%d,%d,%0.4f,%0.4f,%0.4f) %d %0.4f %0.4f %0.4f\n",
	     State->w,
	     State->h,
	     State->r1,
	     State->r2,
	     State->j1,
	     State->j2,
	     State->hv,
	     i,
	     State->spatial_correlation[i],
	     State->joint_entropy[i],
	     State->mutual_information[i]);
    }
    fflush(stdout);
    // Write eps image if needed
    if( epsfn ) {
      io_epsfromstate(State, epsfn, 1.0f, 1.0f);
    }
    exit(0);
  }
}
  

int progress_ca(int **c, int w, int h, int r1, int r2, float j1, float j2, float hv)
{
  int k,i,j,sum1,sum2,ov;

  // We process a single random (i,j) cell at a time
  i = random_rnd(&State->random,State->h);
  j = random_rnd(&State->random,State->w);
  ov = c[i][j];

  // Find sums (from precomputed neighbors)
  sum1 = sum2 = 0;
  for(k=0; k<Neighbors[i][j].nr1nbrs; k++) {
    // Within promotion range
    sum1 += c[Neighbors[i][j].r1nbrs[k].y][Neighbors[i][j].r1nbrs[k].x];
  }
  for(k=0; k<Neighbors[i][j].nr2nbrs; k++) {
    // Within inhibition range
    sum2 += c[Neighbors[i][j].r2nbrs[k].y][Neighbors[i][j].r2nbrs[k].x];
  }
  // Now that sums have been created, used them to find new cell state
  if( (hv + j1*sum1 + j2*sum2) < 0 ) {
    c[i][j] = -1;
  } else {
    c[i][j] = +1;
  }

  // Return the old value
  return (ov==c[i][j]);
}


void tick()
{ 
  static int match_count=0;

  if( State->time == 0 ) {
    match_count = 0;
  }

  if( State->rolling && !(State->converged >= 2) ) {
    // Process one step of the CA
    if( progress_ca(State->cells,
		    State->w,
		    State->h,
		    State->r1,
		    State->r2,
		    State->j1,
		    State->j2,
		    State->hv) ) {
      // No cell change
      match_count++;
    } else {
      // The cell changed
      match_count = 0;
    }
    State->time++;
  }

  /*
  printf("Match count: %d\n",match_count);
  pthread_mutex_unlock(&StateLock);
  UpdateGuiState(((gstate_t*)State));
  pthread_mutex_lock(&StateLock);
  */
  
  // Estiname convergence
  if( match_count > 10000 ) {
    // Convergence has been reached
    if( !State->converged ) {
      handle_convergence();
    }
  }
}

u64b_t get_time()
{
  struct timeval tv;

  gettimeofday(&tv, NULL);

  return tv.tv_sec * 1000000ull + tv.tv_usec;
}

static void simulator_loop()
{
  u64b_t i,t1,t2,sleep;

  if( !State->bch ) {
    // Start the GUI
    StartGUI(VERSION_STR,((gstate_t*)State)); 
    UpdateGuiState(((gstate_t*)State));
  }

  while( 1 ) {
    pthread_mutex_lock(&StateLock);

    // Progress the game one time step
    t1 = get_time();

    for(i=0; i<(State->w+State->h)/2; i++) {
      tick();
    }
    
    pthread_mutex_unlock(&StateLock);

    if( !State->bch ) {
      // Update the gui
      UpdateGuiState(((gstate_t*)State));
    }

    t2 = get_time();

    // Sleep for the remainder of the tick cycle
    if( !State->bch ) {
      // Only in non-batch mode do we slow down for GUI, etc.
      if( (t1 < t2) && ((t2-t1) < 2000) ) {
	sleep = 2000 - (t2-t1);
	if( sleep > 0 ) {
	  usleep( sleep );
	}
      }
    }

  }

}


void print_usage_and_exit(int status)
{
  fprintf(stderr,
	  "Usage: casim2d [OPTION]...\n"
	  "2D Cellular Automata Simulator (c)2012 Aaron Vose and Paul Giblock\n"
	  "\n"
	  "Available Options:\n"
	  "  -r1 RADIUS             Promotional radius            (Default: %d)\n"
	  "  -r2 RADIUS             Inhibitory radius             (Default: %d)\n"
	  "  -j1 SCALE              Promotional scaling           (Default: %f)\n"
	  "  -j2 SCALE              Inhibitory scaling            (Default: %f)\n"
	  "  -hv SCALE              Promotionl constant offset    (Default: %f)\n"
	  "\n"
	  "  -w  CELLS              Width of window               (Default: %d)\n"
	  "  -h  CELLS              Height of window              (Default: %d)\n"
	  "\n"
	  "  -f  FILENAME           EPS file from which to initialize\n"
	  "\n"
	  "  -?, --help            Print the help message and exit\n",
	  PARAM_r1, PARAM_r2, PARAM_j1, PARAM_j2, PARAM_hv, PARAM_w, PARAM_h);

  exit(status);
}
 
 
int parse_int_option(char *str, char *name)
{
  int val;
  
  if (str) {
    if (sscanf(str, "%d", &val) != 1) {
      fprintf(stderr, "-%s requires a valid integer argument.\n\n", name);
      print_usage_and_exit(1);
    }
  } else {
    fprintf(stderr, "-%s option requires an argument.\n\n", name);
    print_usage_and_exit(1);
  }
  return val;
}
 
 
 
float parse_float_option(char *str, char *name)
{
  float val;

  if (str) {
    if (sscanf(str, "%f", &val) != 1) {
      fprintf(stderr, "-%s requires a valid integer argument.\n\n", name);
      print_usage_and_exit(1);
    }
  } else {
    fprintf(stderr, "-%s option requires an argument.\n\n", name);
    print_usage_and_exit(1);
  }

  return val;
}


////////////////////////////////////////////////////////////
// Application entry point
////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
  int a;
  char *opt, *val;

  // Default option
  int   r1  = PARAM_r1;
  int   r2  = PARAM_r2;
  float j1  = PARAM_j1;
  float j2  = PARAM_j2;
  float hv  = PARAM_hv;
  int   w   = PARAM_w;
  int   h   = PARAM_h;
  int   bch = PARAM_batch;

  // Parse inputs
  for(a=1; a<argc; ++a) {
    // Help
    if (strcmp(argv[a], "--help") == 0 || strcmp(argv[a], "-?") == 0) {
      print_usage_and_exit(0);
    }
    // a -X param( 
    else if( ((strlen(argv[a])==2 || strlen(argv[a])==3) || strlen(argv[a])==4) && argv[a][0]=='-') {
      opt = &(argv[a][1]);
      // Only use val if not last argument
      val = (a==argc-1) ? NULL : argv[a+1];
      if( !strcmp("hv", opt) ||
	  !strcmp("HV", opt) ) {
	hv = parse_float_option(val, "hv");
	a++;
      } else if( !strcmp("eps", opt) ||
		 !strcmp("PS", opt) ||
		 !strcmp("ps", opt) ||
		 !strcmp("EPS", opt) ) {
	if( val ) {
	  epsfn = strdup(val);
	}
	a++;
      } else if( !strcmp("B", opt) ||
		 !strcmp("b", opt) ) {
	bch = 1;
      } else if( !strcmp("h", opt) ||
		 !strcmp("H", opt) ) {
	h = parse_int_option(val, "h");
	a++;
      } else if( !strcmp("w", opt) ||
		 !strcmp("W", opt) ) {
	w = parse_int_option(val, "w");
	a++;
      } else if( !strcmp("f", opt) ||
		 !strcmp("F", opt) ) {
	// Input EPS file
	State = io_epstostate(val);
	a++;
      } else if( !strcmp("r1", opt) ||
		 !strcmp("R1", opt) ) {
	r1 = parse_int_option(val, "r1");
	a++;
      } else if( 
		!strcmp("r2", opt) ||
		!strcmp("R2", opt) ) {
	r2 = parse_int_option(val, "r2");
	a++;
      } else if( !strcmp("j1", opt) ||
		 !strcmp("J1", opt) ) {
	j1 = parse_float_option(val, "j1"); 
	a++;
      } else if( !strcmp("j2", opt) ||
		 !strcmp("J2", opt) ) {
	j2 = parse_float_option(val, "j2");
	a++;
      } else {
	fprintf(stderr, "Unknown option '-%s'.\n\n", opt);
	print_usage_and_exit(1);
      }
    }
  }
  
  // Get ready to do stuffs...
  if( !State ) {
    // The state may have already been set from a file.
    init_state(w, h, r1, r2, j1, j2, hv, bch);
  }

  // Compute neighbor array
  Compute_Neighbors();

  // Game loop
  simulator_loop();

  // Done, exit game.
  return 0;
}

