#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ps.h"
#include "casim2d.h"
#define IO_EPS_C
#include "io_eps2d.h"

static void* Malloc(size_t size)
{
  void *ptr;

  if( !(ptr=malloc(size)) ) {
    fprintf(stderr,"Failed to allocate space while reading in an EPS file.\n");
    exit(1);
  }

  return ptr;
}

////////////////////////////////////////////////////////////

state_t* io_epstostate(char *fn)
{
  FILE    *f;
  state_t *state;
  char     line[8192],*t;
  float    hv,j1,j2;
  int      i,r1,r2,x,y,v,w,h;
  struct timeval tv;

  // Open EPS file
  if( !(f=fopen(fn,"r")) ) {
    return NULL;
  }

  // Allocate a new state object
  state = Malloc(sizeof(state_t));
  memset(state, 0, sizeof(state_t));
 
  // Process one line at a time
  while( fgets(line,sizeof(line),f) ) {
    if( (t=strstr(line,"casim2d_hdr:")) ) {
      // Header line
      // !!av: evil hack to get around the fack that parent uses /runs as the holding dir...
      // casim2d_hdr:runs/w_30_h_30_r1_1_r2_6_h_0_j1_0_j2_-0.1__0.ep
      //%% casim2d_hdr:w_128_h_128_r1_1_r2_6_hv_0.000000_j1_1.000000_j2_0.100000__0.eps
      for(i=strlen(line); i>=0; i--) {
	if( (line[i] == '/') || (line[i] == ':') ) {
	  i++;
	  t = &(line[i]);
	  break;
	}
      }
      if( sscanf(t,"w_%d_h_%d_r1_%d_r2_%d_hv_%f_j1_%f_j2_%f",
		 &w,&h,&r1,&r2,&hv,&j1,&j2) != 7 ) {
	fprintf(stderr,"Failed to parse EPS header line.\n");
	return NULL;
      }
      state->r1 = r1;
      state->r2 = r2;
      state->j1 = j1;
      state->j2 = j2;
      state->hv = hv;
      state->w = w;
      state->h = h;
      state->cellsx = w;
      state->cellsy = h;
      state->cells = Malloc(h*sizeof(int*));
      for(i=0; i<h; i++) {
	state->cells[i] = Malloc(w*sizeof(int));
      }
    } else if( (t=strstr(line,"casim2d_cell:")) ) {
      // Cell line
      if( !state->cells ) {
	fprintf(stderr,"Failed to find EPS header line before cell line.\n");
	return NULL;
      }
      // Fill in the cell
      if( sscanf(t+strlen("casim2d_cell:"),"%d,%d:%d",&x,&y,&v) != 3 ) {
	fprintf(stderr,"Failed to read EPS cell line.\n");
	return NULL;
      }
      state->cells[y][x] = v;
    }
  }

  // Assume image time is ... 0?
  state->time = 0;

  // Initialize the random number generator
  if( gettimeofday(&tv, NULL) == -1 ) {
    random_initrand(&state->random,7);
  } else {
    random_initrand(&state->random,tv.tv_usec);
  }
  
  // Return the newly created state
  return state;
}


// Write the state out
void io_epsfromstate(state_t *st, char *fn, float w, float h)
{
  static int  ss=0;
  char        fnd[1024],buf[1024];
  FILE       *f;
  int         i,j;
  
  // RLE
  int lastc;
  int lastj;
  int c = 0;
  
  int cx = (float)st->cellsx;
  int cy = (float)st->cellsy;
  
  if( !fn ) {
    // Generate a filename from the rule table
    fn = fnd;
    sprintf(fn,"w_%d_h_%d_r1_%d_r2_%d_hv_%f_j1_%f_j2_%f",
	    st->w,  st->h,
	    st->r1, st->r2,
	    st->hv,
	    st->j1, st->j2
	    );
    sprintf(buf,"__%d.eps",ss);
    ss++;
    strcat(fn, buf);
  }
  
  // Renter EPS
  f = PS_Start(fn, w, h);
  sprintf(buf,"casim2d_hdr:%s",fn);
  PS_Ccomment(f, buf);
  PS_Color(f, 
	   0.0f,
	   0.0f, 
	   0.0f);
  PS_Quad(f,
	  0.0, 0.0,
	  w,   0.0,
	  w,   h,
	  0.0, h,
	  1);
  
  for(i=0; i<cy; i++) {
    // Initialize first run
    lastj = 0;
    lastc = st->cells[i][0];
    
    for(j=0; j<=cx; j++) {
      // Adopt color if not past last cell
      if ( j != cx ) {
	c = st->cells[i][j];
	// Always print comment for any live cell
	sprintf(buf,"casim2d_cell:%d,%d:%d", j, i, c);
	PS_Ccomment(f, buf);
      }
      // Draw last run if change or EOL
      if ( (c != lastc) || (j == cx) ) {
	if( lastc == 1 ) {
	  PS_Color(f,1.0f,0.0f,0.0f);
	} else if( lastc == -1 ) {
	  PS_Color(f,0.0f,0.0f,1.0f);
	} else {
	  PS_Color(f,0.0f,1.0f,0.0f);
	}
	PS_Quad(f,
		((float)lastj)/cx*w, ((float)i)  /cy*h,
		((float)lastj)/cx*w, ((float)i+1)/cy*h,
		((float)j)/cx*w,     ((float)i+1)/cy*h,
		((float)j)/cx*w,     ((float)i)  /cy*h,
		1);
	lastc = c;
	lastj = j;
      }
    }
  }
  
  PS_End(f);
}

