#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

#include "types.h"
#include "util.h"
#include "random.h"
#include "params.h"
#include "io_eps1d.h"
#define ONEDCASIM_C
#include "casim1d.h"
#include "gui.h"


static state_t  *State = NULL;
pthread_mutex_t  StateLock = PTHREAD_MUTEX_INITIALIZER;

static int num_rules(int r, int K)
{
  return (2*r+1)*(K-1) + 1;
}

////////////////////////////////////////////////////////////
// Sets up the global state to get ready for play
////////////////////////////////////////////////////////////
static void init_state(int r, int K, int* rules, int w, int h, int l)
{
  struct timeval tv;
  int            i;
  
  // Allocate the main state structure
  if( !(State=malloc(sizeof(state_t))) ) {
    Error("Could not allocate state structure (%u)\n",sizeof(state_t));
  }

  // Go ahead and clear the memory to start with.
  memset(State,0,sizeof(state_t));

  // Initialize the random number generator
  if( gettimeofday(&tv, NULL) == -1 ) {
    Warn("gettiemofday() failed; seeding random generator with 7 instead.\n");
    random_initrand(&State->random,7);
  } else {
    random_initrand(&State->random,tv.tv_usec);
  }

  // Set the current time
  State->time = 0;

  State->rolling = 0;

  // Allocate cells
  State->cellsx = w;
  State->cellsy = h;
  State->cells = malloc(State->cellsy*sizeof(int*));
  for(i=0; i<State->cellsy; i++) {
    State->cells[i] = malloc(State->cellsx*sizeof(int));
    memset(State->cells[i], 0, State->cellsx*sizeof(int));
  }  
  // Give a random starting state
  for(i=0; i<State->cellsx; i++) {
    State->cells[0][i] = random_U(&State->random,K);
  }

  // Initialize history
  State->history = history_create(State->cells[0], State->cellsx, NULL);
	State->c_tick   = 0;
  State->c_class  = 0;
  State->c_cycle  = 0;
  State->c_stride = 0;

  // Setup rules
  State->nrules = num_rules(r, K);
  if (rules) {
    State->rules = rules;
  } else {
    State->rules = malloc(State->nrules*sizeof(int));
    for(i=0; i< State->nrules; i++) {
      if( !i ) {
        State->rules[i] = 0;
      } else {
        State->rules[i] = 1+random_U(&State->random,K-1);
      }
    }
  }

  State->r = r;
  State->K = K;
	State->nhistory = l;
}


void set_state(int setup_rules)
{
  int i;
 
  pthread_mutex_lock(&StateLock);  

  // Setup some rules for processing the CA
  if( setup_rules ) {
    for(i=0; i< State->nrules; i++) {
      if( !i ) {
        State->rules[i] = 0;
      } else {
        State->rules[i] = 1+random_U(&State->random,State->K-1);
      }
    }
  }

  // Give a random starting state
  for(i=0; i<State->cellsx; i++) {
    State->cells[0][i] = random_U(&State->random,State->K);
  }

  // Clear the rest of the board
  for(i=1; i<State->cellsy; i++) {
    memset(State->cells[i], 0, State->cellsx*sizeof(int));
  }

  // Set the current time
  State->time = 0;

  // Reset history
  history_free(State->history);
  State->history = history_create(State->cells[0], State->cellsx, NULL);
  State->c_tick   = 0;
  State->c_class  = 0;
  State->c_cycle  = 0;
  State->c_stride = 0;

  pthread_mutex_unlock(&StateLock);
}


void decimate()
{
  int i,s;

  pthread_mutex_lock(&StateLock);  

  for(i=s=0; i<State->nrules; i++) {
    s += State->rules[i];
  }
  if( s ) {
    while( !State->rules[i=random_U(&State->random,State->nrules)] );
    State->rules[i] = 0;
  }

  pthread_mutex_unlock(&StateLock);
}


int run_state()
{
  pthread_mutex_lock(&StateLock);  
  State->rolling = !State->rolling;
  pthread_mutex_unlock(&StateLock);
  return State->rolling;
}



////////////////////////////////////////////////////////////
// Main game loop and helpers
////////////////////////////////////////////////////////////

static int loop_index(int i)
{
  // Catch the case where i is too small
  while( i < 0 ) {
    i = State->cellsx + i;
  }

  // Catch the case where i is too large
  while( i >= State->cellsx ) {
    i = i - State->cellsx;
  }

  // Return the adjusted index
  return i;
}


void progress_ca(int *c0, int *c1, int *rules, int ncells, int radius)
{
  int i,j,sum;

  for(i=0; i<ncells; i++) {
    // Find the sum for cell i based on neighbords
    sum = 0;
    for(j=-State->r; j<=State->r; j++) {
      sum += c0[loop_index(i+j)];      
    }
    // Apply rule
    c1[i] = rules[sum];
  }
}


void tick_history()
{
	State->c_tick++;

  // Check in history
  history_t *h, *eh = 0;
  int age, hc;

  // Previous history line
  h = State->history->prev;

  if (State->c_class != 0 || !h ) {
    // Already found class, or no history, skip
    return;
  }

  // Check previous line first for Class I
  if (history_compare(State->history, h, State->cellsx) >= 0) {
    State->c_class  = 1;
    State->c_cycle  = 1;
    State->c_stride = 0;
    printf("Class I detected, %ld transient\n", State->c_tick - 1);
    fflush(stdout);
    return;
  }

  // Now check all older history for Class II
  for (age = 1; h; h=h->prev, age++) {
    if ((hc = history_compare(State->history, h, State->cellsx)) >= 0) {
      State->c_class  = 2;
      State->c_cycle  = age;
      State->c_stride = hc;
      printf("Class II detected, %ld transient, %d period, %d stride\n", State->c_tick - age, age, hc);
      fflush(stdout);
      return;
    }

    // This entry is the new end, we will delete it stuff after it later
    if (age == State->nhistory) {
      eh = h;
    }
  }
  
  // Trim slack
  if (eh) {
    history_free(eh->prev);
    eh->prev = NULL;
  }
}




void tick()
{ 
  int *c0, *c1;

  if( State->time < (State->cellsy-1) ) {
    c0 = State->cells[State->time];
    c1 = State->cells[State->time+1];
  } else if( State->rolling ) {
	// Shift rows
    int  botidx= State->cellsy-1;
    int *top   = State->cells[0];
    memcpy(&State->cells[0], &State->cells[1], botidx*sizeof(int*));
    State->cells[botidx] = top;

    // Process the new bottom..
    c0 = State->cells[botidx-1];
    c1 = State->cells[botidx];
  } else {
    // Not progressing
    return;
  }

  // Process one step of the CA
  progress_ca(c0, c1,
              State->rules,
              State->cellsx,
              State->r);

  // Store in history
  State->history = history_create(c1, State->cellsx, State->history);

  // Update Class state based on history
  tick_history();
}

u64b_t get_time()
{
  struct timeval tv;

  gettimeofday(&tv, NULL);

  return tv.tv_sec * 1000000ull + tv.tv_usec;
}

static void game_loop()
{
  u64b_t t1,t2,sleep;
  int    i,speed=1;

  // Start the GUI
  StartGUI("v"VERSION_STR,((gstate_t*)State)); 
  UpdateGuiState(((gstate_t*)State));

  while( 1 ) {
    pthread_mutex_lock(&StateLock);

    // Progress the game one time step
    t1 = get_time();

    // Update ticks / time
    for(i=0; i<speed; i++) {
      tick();
      if( State->time < (State->cellsy-1) ) {
	State->time++;
      }
    }
    
    pthread_mutex_unlock(&StateLock);

    // Update the gui
    UpdateGuiState(((gstate_t*)State));
    t2 = get_time();

    // Sleep for the remainder of the tick cycle
    if( (t1 < t2) && ((t2-t1) < 10000) ) {
      sleep = 10000 - (t2-t1);
      if( sleep > 0 ) {
        usleep( sleep );
      }
    }
  }

}


void print_usage_and_exit(int status)
{
  fprintf(stderr,
      "Usage: casim1d [OPTION]...\n"
      "1D Cellular Automata Simulator (c) 2012 Aaron Vose and Paul Giblock\n"
      "\n"
      "Available Options:\n"
      "  -r RADIUS             Neighborhoood radius            (Default: %d)\n"
      "  -K STATES             Number of states per cell       (Default: %d)\n"
      "  -t RULE1,RULE2,...    Initial rule table\n"
      "\n"
      "  -w CELLS              Width of window                 (Default: %d)\n"
      "  -h CELLS              Height of window                (Default: %d)\n"
      "  -l STATES             Class-detection history entries (Default: %d)\n"
      "\n"
      "  -f FILENAME           EPS file from which to initialize\n"
      "\n"
      "  -?, --help            Print the help message and exit\n"
      "  -v, --version         Print the version number and exit\n",
      PARAM_r, PARAM_K, PARAM_w, PARAM_h, PARAM_HISTORY);
  exit(status);
}


int parse_int_option(char* str, char name) {
  int val;
  if (str) {
    if (sscanf(str, "%d", &val) != 1) {
      fprintf(stderr, "-%c requires a valid integer argument.\n\n", name);
      print_usage_and_exit(1);
    }
  } else {
    fprintf(stderr, "-%c option requires an argument.\n\n", name);
    print_usage_and_exit(1);
  }
  return val;
}


////////////////////////////////////////////////////////////
// Application entry point
////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
  int a, i;
  char opt, *val, *rule;

  // Parsed option
  int *rules = NULL;
  int  nrules = num_rules(PARAM_r, PARAM_K);
  int  r = PARAM_r;
  int  K = PARAM_K; 
  int  w = PARAM_w;
  int  h = PARAM_h;
  int  l = PARAM_HISTORY;

  // Parse inputs
  for(a=1; a<argc; ++a) {
    // Help
    if (strcmp(argv[a], "--help") == 0 || strcmp(argv[a], "-?") == 0) {
      print_usage_and_exit(0);
    }
    // Version number
		else if (strcmp(argv[a], "--version") == 0 || strcmp(argv[a], "-v") == 0) {
      printf("1D Cellular Automata Simulator %s\n", VERSION_STR);
			exit(0);
    }
    // a -X param 
    else if (strlen(argv[a])==2 && argv[a][0]=='-') {
      opt = argv[a][1];
      // Only use val if not last argument, and if next argument isn't -
      val = (a==argc-1) || argv[a+1][0]=='-' ? NULL : argv[a+1];

      switch(opt) {
        case 'h':
        case 'H':
          h = parse_int_option(val, 'h');
          break;
        case 'w': 
        case 'W':
          w = parse_int_option(val, 'w');
          break;
        case 'f': 
        case 'F':     
          // Input EPS file
          State = io_epstostate(val);
          break;
        case 'r': // Radius
        case 'R':
          r = parse_int_option(val, 'r');
          break;
        case 'k':
        case 'K':
          K = parse_int_option(val, 'K');
          break;
        case 'l':
        case 'L':
          l = parse_int_option(val, 'L');
          break;
        case 't': // Table
        case 'T':
          if (val) {
            // Count commas, get length
            nrules=1;
            for(i=0; i<strlen(val); ++i) {
              if (val[i] == ',') { nrules++; }
            }

            // TODO: check for errors
            rules = malloc(nrules*sizeof(int));
            rule  = strtok(val, ",");
            for(i=0; i<nrules && rule; ++i) {
              if (sscanf(rule, "%d", &rules[i]) != 1) {
                fprintf(stderr, "Rule %d with value `%s' is invalid.\n\n", i+1, rule);
                print_usage_and_exit(1);
              }
              rules[i] = atoi(rule);
              rule  = strtok(NULL, ",");
            }
          } else {
            fprintf(stderr, "Missing argument to -t.\n\n");
            print_usage_and_exit(1);
          }

          break;

        default:
          fprintf(stderr, "Unknown option '-%c'.\n\n", opt);
          print_usage_and_exit(1);
      }
    }
  }

  if (rules && (nrules != num_rules(r, K))) {
      fprintf(stderr, "Rules table must be size %d.\n\n", num_rules(r, K));
      print_usage_and_exit(1);
  }

  // Get ready to do stuffs...
  if( !State ) {
    // The state may have already been set from a file.
    init_state(r, K, rules, w, h, l);
  } else {
    // Se sure starting classification is found
    for(i=0; i<State->time; i++) {
      State->history = history_create(State->cells[i], State->cellsx, State->history);
    }
    tick_history();
  }
      
  // Game loop
  game_loop();

  // Done, exit game.
  return 0;
}

